namespace App\Services;

use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use Kreait\Firebase\Exception\Messaging\NotFound;

class PushNotificationService
{
    protected $messaging;

    public function __construct()
    {
        $this->messaging = (new Factory)
            ->withServiceAccount([
                "type" => "service_account",
                "project_id" => env('FIREBASE_PROJECT_ID'),
                "private_key_id" => env('FIREBASE_PRIVATE_KEY_ID'),
                "private_key" => str_replace("\\n", "\n", env('FIREBASE_PRIVATE_KEY')),
                "client_email" => env('FIREBASE_CLIENT_EMAIL'),
                "client_id" => env('FIREBASE_CLIENT_ID'),
                "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
                "token_uri" => "https://oauth2.googleapis.com/token",
                "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
                "client_x509_cert_url" => env('FIREBASE_CLIENT_X509_CERT_URL'),
                "universe_domain" => "googleapis.com"
            ])
            ->createMessaging();
    }

    public function sendToUsers($users, $title, $body)
    {
        $success = 0;
        $fail = 0;

        foreach ($users as $user) {
            $tokens = $user->deviceTokens()->pluck('token');

            foreach ($tokens as $token) {
                try {
                    $notification = Notification::create($title, $body);
                    $message = CloudMessage::withTarget('token', $token)
                        ->withNotification($notification)
                        ->withHighestPossiblePriority()
                        ->withDefaultSounds();

                    $this->messaging->send($message);
                    Log::channel('log_notifications')->info("Sent to {$user->name} - Token: {$token}");
                    $success++;
                } catch (NotFound $e) {
                    $user->deviceTokens()->where('token', $token)->delete();
                    Log::channel('log_notifications')->warning("Token not found for {$user->name}: {$token}");
                    $fail++;
                } catch (\Exception $e) {
                    Log::channel('log_notifications')->error("Error for {$user->name}: " . $e->getMessage());
                    $fail++;
                }
            }
        }

        return ['success' => $success, 'fail' => $fail];
    }
}
