<?php

namespace App\Exports;

use App\Models\SendSms;
use App\Models\Classroom;
use App\Models\User;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class SmsExport implements FromCollection, WithHeadings, ShouldAutoSize, WithStyles, WithTitle
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    public function collection()
    {
        $query = SendSms::query();

        // Aplicar filtros
        if (!empty($this->filters['search'])) {
            $query->where(function($q) {
                $q->where('title', 'like', '%' . $this->filters['search'] . '%')
                  ->orWhere('type', 'like', '%' . $this->filters['search'] . '%');
            });
        }

        if (isset($this->filters['type']) && $this->filters['type'] !== '') {
            $query->where('type', $this->filters['type']);
        }

        if (!empty($this->filters['class'])) {
            $query->where('class_id', $this->filters['class']);
        }

        $sms = $query->latest()->get();

        return $sms->map(function ($model) {
            $classroom = !empty($model->class_id) ? Classroom::find($model->class_id) : null;
            $student = !empty($model->student_id) ? User::find($model->student_id) : null;

            $typeLabel = '';
            if($model->type == "all") {
                $typeLabel = "Todos";
            } elseif($model->type == "student") {
                $typeLabel = "Estudante";
            } elseif($model->type == "class") {
                $typeLabel = "Classe";
            }

            return [
                'para' => $typeLabel,
                'classe_turma' => $classroom ? $classroom->class . 'ª Classe - ' . $classroom->name : '-',
                'estudante' => $student ? $student->name : '-',
                'descricao' => strip_tags($model->title),
                'ficheiro' => $model->media_name ?: '-',
                'data' => $model->created_at->format('d/m/Y H:i'),
            ];
        });
    }

    public function headings(): array
    {
        return [
            'Para',
            'Classe/Turma',
            'Estudante',
            'Descrição',
            'Ficheiro',
            'Data'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Estilo do cabeçalho
        $sheet->getStyle('A1:F1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
                'size' => 12,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4e73df'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ]);

        // Estilo das linhas de dados
        $lastRow = $sheet->getHighestRow();
        if ($lastRow > 1) {
            $sheet->getStyle('A2:F' . $lastRow)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => 'DDDDDD'],
                    ],
                ],
                'alignment' => [
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ]);

            // Zebrar as linhas
            for ($row = 2; $row <= $lastRow; $row++) {
                if ($row % 2 == 0) {
                    $sheet->getStyle('A' . $row . ':F' . $row)->applyFromArray([
                        'fill' => [
                            'fillType' => Fill::FILL_SOLID,
                            'startColor' => ['rgb' => 'F8F9FA'],
                        ],
                    ]);
                }
            }
        }

        // Altura da primeira linha
        $sheet->getRowDimension(1)->setRowHeight(25);

        return [];
    }

    public function title(): string
    {
        return 'Lista de SMS';
    }
}
