<?php

namespace App\Http\Controllers;

use App\Models\PaymentReference;
use App\Models\Fee_assign;
use App\Models\FeeStructure;
use App\Services\FeeCalculationService;
use App\Services\PaymentReceiptService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class FeeApiController extends Controller
{
    protected $feeCalculationService;
    protected $receiptService;

    public function __construct(FeeCalculationService $feeCalculationService, PaymentReceiptService $receiptService)
    {
        $this->feeCalculationService = $feeCalculationService;
        $this->receiptService = $receiptService;
    }

    /**
     * Get student fee summary for current year
     * GET /api/fees/summary
     */
    public function getSummary(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $year = $request->input('year', date('Y'));

            // Usar serviço para calcular resumo do ano
            $yearSummary = $this->feeCalculationService->calculateYearSummary($user, $year);

            return response()->json([
                'status' => 'true',
                'data' => [
                    'student' => [
                        'id' => $user->id,
                        'student_id' => $user->student_id,
                        'name' => $user->name,
                        'class' => $user->class->class ?? 'N/A',
                        'classroom' => $user->classroom->name ?? 'N/A',
                    ],
                    'year' => $year,
                    'summary' => [
                        'total_expected' => $yearSummary['total_expected'],
                        'total_paid' => $yearSummary['total_paid'],
                        'total_pending' => $yearSummary['total_pending'],
                        'total_overdue' => $yearSummary['total_overdue'],
                        'total_fines' => $yearSummary['total_fines'],
                        'total_discounts' => $yearSummary['total_discounts'],
                    ],
                    'payment_progress' => [
                        'months_total' => 12,
                        'months_paid' => count(array_filter($yearSummary['months_detail'], fn($m) => $m['is_paid'])),
                        'percentage' => $yearSummary['total_expected'] > 0
                            ? round(($yearSummary['total_paid'] / $yearSummary['total_expected']) * 100, 1)
                            : 0,
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error in fee summary API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error fetching fee summary',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get detailed monthly fee breakdown
     * GET /api/fees/monthly
     */
    public function getMonthlyBreakdown(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $year = $request->input('year', date('Y'));

            $yearSummary = $this->feeCalculationService->calculateYearSummary($user, $year);
            $monthlyData = $yearSummary['months_detail'];

            // Formatar dados mensais para mobile
            $formattedMonths = array_map(function($month) {
                return [
                    'month' => $month['month'],
                    'year' => $month['year'],
                    'base_amount' => (float) $month['base_amount'],
                    'fine_amount' => (float) $month['fine_amount'],
                    'discount_amount' => (float) ($month['discount_applied'] ?? 0),
                    'total_expected' => (float) $month['total_expected'],
                    'paid_amount' => $month['is_paid'] ? (float) ($month['paid_amount'] ?? $month['total_expected']) : 0,
                    'is_paid' => (bool) $month['is_paid'],
                    'is_overdue' => (bool) $month['is_overdue'],
                    'due_date' => $month['due_date'] ? $month['due_date']->format('Y-m-d') : null,
                    'payment_date' => $month['is_paid'] && isset($month['payment_date'])
                        ? (is_string($month['payment_date']) ? $month['payment_date'] : $month['payment_date']->format('Y-m-d H:i:s'))
                        : null,
                    'status' => $month['is_paid'] ? 'paid' : ($month['is_overdue'] ? 'overdue' : 'pending'),
                    'fee_names' => $month['fee_names'] ?? 'Taxa Escolar',
                ];
            }, $monthlyData);

            return response()->json([
                'status' => 'true',
                'data' => [
                    'year' => $year,
                    'months' => $formattedMonths
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error in monthly breakdown API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error fetching monthly breakdown',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment references for student
     * GET /api/fees/references
     */
    public function getPaymentReferences(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $year = $request->input('year', date('Y'));
            $status = $request->input('status'); // paid, pending, expired

            $query = PaymentReference::where('student_id', $user->id)
                ->where('fee_year', $year);

            if ($status) {
                $query->where('status', $status);
            }

            $references = $query->orderBy('created_at', 'desc')->get();

            $formattedReferences = $references->map(function($ref) {
                return [
                    'id' => $ref->id,
                    'entity_code' => $ref->entity_code,
                    'reference_number' => $ref->reference_number,
                    'reference_formatted' => $this->formatReference($ref->reference_number),
                    'amount' => (float) $ref->amount,
                    'fee_month' => $ref->fee_month,
                    'fee_year' => $ref->fee_year,
                    'status' => $ref->status,
                    'expires_at' => $ref->expires_at ? Carbon::parse($ref->expires_at)->format('Y-m-d H:i:s') : null,
                    'paid_at' => $ref->paid_at ? Carbon::parse($ref->paid_at)->format('Y-m-d H:i:s') : null,
                    'created_at' => $ref->created_at->format('Y-m-d H:i:s'),
                    'is_expired' => $ref->expires_at && Carbon::parse($ref->expires_at)->isPast() && $ref->status === 'pending',
                    'days_until_expiry' => $ref->expires_at && $ref->status === 'pending'
                        ? now()->diffInDays(Carbon::parse($ref->expires_at), false)
                        : null,
                ];
            });

            return response()->json([
                'status' => 'true',
                'data' => $formattedReferences
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error in payment references API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error fetching payment references',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate payment reference
     * POST /api/fees/generate-reference
     */
    public function generateReference(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'month' => 'required|string',
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'false',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            // Verificar se já existe referência pendente
            $existingReference = PaymentReference::where([
                'student_id' => $user->id,
                'fee_month' => $request->month,
                'fee_year' => $request->year,
                'status' => 'pending'
            ])->first();

            if ($existingReference) {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Já existe uma referência pendente para ' . $request->month . '/' . $request->year,
                    'existing_reference' => [
                        'id' => $existingReference->id,
                        'reference_number' => $existingReference->reference_number,
                        'entity_code' => $existingReference->entity_code,
                        'amount' => (float) $existingReference->amount,
                        'expires_at' => $existingReference->expires_at
                            ? Carbon::parse($existingReference->expires_at)->format('Y-m-d H:i:s')
                            : null,
                    ]
                ], 400);
            }

            // Obter cálculo de fee para o mês
            $calculation = $this->feeCalculationService->calculateMonthFee($user, $request->month, $request->year);

            if (!$calculation) {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Não foi possível calcular o valor da taxa para este mês'
                ], 400);
            }

            // Calcular valores
            $baseAmount = $calculation['base_amount'];
            $fineAmount = $calculation['late_payment_fine'] ?? 0;
            $discount = $request->input('discount', 0);
            $totalAmount = $baseAmount + $fineAmount - $discount;

            // Configurações
            $entity = config('payments.entity', '11111');
            $ttlDays = (int) config('payments.reference_ttl_days', 3);

            // Calcular data de expiração baseada na due_date
            $expiresAt = $calculation['due_date'] ?? now()->addDays($ttlDays);
            if ($expiresAt->isPast()) {
                $expiresAt = now()->addDays($ttlDays);
            }
            $expiresAt = $expiresAt->endOfDay();

            // Gerar referência única
            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
            $reference = $referenceGenerator->makeFromStudent(
                $user,
                $request->month,
                $request->year,
                $totalAmount
            );

            // Criar registro de payment reference
            $paymentReference = PaymentReference::create([
                'student_id' => $user->id,
                'entity_code' => $entity,
                'reference_number' => $reference,
                'amount' => $totalAmount,
                'fine_amount' => $fineAmount,
                'fee_month' => $request->month,
                'fee_year' => $request->year,
                'expires_at' => $expiresAt,
                'status' => 'pending'
            ]);

            Log::info('Payment reference generated via API', [
                'user_id' => $user->id,
                'reference' => $reference,
                'month' => $request->month,
                'year' => $request->year,
                'amount' => $totalAmount
            ]);

            return response()->json([
                'status' => 'true',
                'message' => 'Referência gerada com sucesso',
                'data' => [
                    'id' => $paymentReference->id,
                    'entity_code' => $entity,
                    'reference_number' => $reference,
                    'reference_formatted' => $this->formatReference($reference),
                    'amount' => (float) $totalAmount,
                    'base_amount' => (float) $baseAmount,
                    'fine_amount' => (float) $fineAmount,
                    'discount_amount' => (float) $discount,
                    'fee_month' => $request->month,
                    'fee_year' => $request->year,
                    'status' => 'pending',
                    'expires_at' => $expiresAt->format('Y-m-d H:i:s'),
                    'days_until_expiry' => now()->diffInDays($expiresAt, false),
                    'payment_instructions' => [
                        'mpesa' => "Menu → Pagar Conta → Entidade: {$entity} → Referência: {$reference}",
                        'emola' => "Pagar Serviços → Entidade: {$entity} → Referência: {$reference}",
                        'atm' => "Pagar Serviços → Entidade: {$entity} → Referência: {$reference}",
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error generating payment reference API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error generating payment reference',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download payment receipt (comprovativo)
     * GET /api/fees/download-receipt/{paymentId}
     */
    public function downloadReceipt($paymentId)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $payment = PaymentReference::findOrFail($paymentId);

            // Verificar se pertence ao estudante
            if ($payment->student_id !== $user->id) {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Acesso negado a este comprovativo'
                ], 403);
            }

            // Verificar se está pago
            if ($payment->status !== 'paid') {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Comprovativo disponível apenas para pagamentos confirmados'
                ], 400);
            }

            $result = $this->receiptService->downloadReceipt($payment);

            if ($result['success']) {
                return response()->json([
                    'status' => 'true',
                    'message' => 'Comprovativo gerado com sucesso',
                    'data' => [
                        'url' => $result['url'],
                        'path' => $result['path'],
                        'type' => 'comprovativo'
                    ]
                ], 200);
            } else {
                return response()->json([
                    'status' => 'false',
                    'message' => $result['message'] ?? 'Erro ao gerar comprovativo'
                ], 500);
            }

        } catch (\Exception $e) {
            Log::error('Error downloading receipt API', [
                'user_id' => $user->id,
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error downloading receipt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download official receipt (recibo oficial)
     * GET /api/fees/download-official-receipt/{paymentId}
     */
    public function downloadOfficialReceipt($paymentId)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $payment = PaymentReference::findOrFail($paymentId);

            // Verificar se pertence ao estudante
            if ($payment->student_id !== $user->id) {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Acesso negado a este recibo'
                ], 403);
            }

            // Verificar se está pago
            if ($payment->status !== 'paid') {
                return response()->json([
                    'status' => 'false',
                    'message' => 'Recibo disponível apenas para pagamentos confirmados'
                ], 400);
            }

            $result = $this->receiptService->downloadOfficialReceipt($payment);

            if ($result['success']) {
                return response()->json([
                    'status' => 'true',
                    'message' => 'Recibo oficial gerado com sucesso',
                    'data' => [
                        'url' => $result['url'],
                        'path' => $result['path'],
                        'type' => 'recibo_oficial'
                    ]
                ], 200);
            } else {
                return response()->json([
                    'status' => 'false',
                    'message' => $result['message'] ?? 'Erro ao gerar recibo oficial'
                ], 500);
            }

        } catch (\Exception $e) {
            Log::error('Error downloading official receipt API', [
                'user_id' => $user->id,
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error downloading official receipt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment history
     * GET /api/fees/payment-history
     */
    public function getPaymentHistory(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $year = $request->input('year');
            $limit = $request->input('limit', 50);

            $query = PaymentReference::where('student_id', $user->id)
                ->where('status', 'paid')
                ->orderBy('paid_at', 'desc');

            if ($year) {
                $query->where('fee_year', $year);
            }

            $payments = $query->limit($limit)->get();

            $formattedPayments = $payments->map(function($payment) {
                return [
                    'id' => $payment->id,
                    'reference_number' => $payment->reference_number,
                    'amount' => (float) $payment->amount,
                    'fee_month' => $payment->fee_month,
                    'fee_year' => $payment->fee_year,
                    'paid_at' => $payment->paid_at ? Carbon::parse($payment->paid_at)->format('Y-m-d H:i:s') : null,
                    'payment_method' => 'Pagamento via Referência',
                ];
            });

            return response()->json([
                'status' => 'true',
                'data' => [
                    'total' => $payments->count(),
                    'payments' => $formattedPayments
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error in payment history API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error fetching payment history',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper: Format reference number
     */
    private function formatReference($reference)
    {
        $ref = preg_replace('/\D/', '', $reference);

        if (strlen($ref) === 11) {
            return substr($ref, 0, 3) . ' ' . substr($ref, 3, 3) . ' ' . substr($ref, 6, 3) . ' ' . substr($ref, 9, 2);
        }

        return $reference;
    }
}
