<?php

namespace App\Http\Controllers;

use App\Models\PaymentConfirmation;
use App\Models\PaymentReference;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use LaravelMultipleGuards\Traits\FindGuard;

class StudentStatementController extends Controller
{
    use FindGuard;

    /**
     * Gerar extrato completo de pagamentos do estudante em PDF
     */
    public function downloadStatement(Request $request)
    {
        // Aumentar tempo de execução para 60 segundos (geração de PDF pode ser lenta)
        set_time_limit(60);

        // Se student_id for fornecido (admin acessando), usar esse student_id
        // Senão, usar o usuário autenticado
        if ($request->has('student_id')) {
            $user = User::find($request->student_id);

            if (!$user) {
                return redirect()->back()->with('error', 'Estudante não encontrado');
            }
        } else {
            $user = $this->findGuardType()->user();

            if (!$user) {
                return redirect()->back()->with('error', 'Usuário não autenticado');
            }
        }

        // Buscar informações da turma
        $classroom = \App\Models\Classroom::find($user->classroom_id);

        // Ano atual
        $currentYear = now()->year;

        // Meses do ano em PORTUGUÊS (para conversão de número para nome)
        $monthNames = ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
                      'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];

        // Buscar TODOS os pagamentos do estudante (fee_assign - Dinheiro, POS, etc)
        $feePayments = \App\Models\Fee_assign::where('student_id', $user->id)
            ->where('year', $currentYear)
            ->get();

        // Buscar referências de pagamento (Entidade e Referência)
        $paymentReferences = PaymentReference::where('student_id', $user->id)
            ->where('fee_year', $currentYear)
            ->get();

        // Array para armazenar dados de pagamento (APENAS MESES ATRIBUÍDOS)
        $allMonthsData = [];
        $totalPaid = 0;
        $totalPending = 0;
        $totalExpired = 0;

        // Processar pagamentos do fee_assign
        foreach ($feePayments as $payment) {
            $monthName = $payment->month;

            // Encontrar o número do mês
            $monthNumber = array_search($monthName, $monthNames) + 1;
            if ($monthNumber === false) {
                continue; // Pular se o nome do mês não for reconhecido
            }

            $totalAmount = ($payment->amount ?? 0) + ($payment->fine ?? 0) - ($payment->discount ?? 0);

            $paymentData = [
                'month_number' => $monthNumber,
                'month_name' => $monthName,
                'amount' => $totalAmount,
                'base_amount' => $payment->amount ?? 0,
                'fine' => $payment->fine ?? 0,
                'discount' => $payment->discount ?? 0,
                'status' => strtolower($payment->status) === 'paid' ? 'paid' : 'pending',
                'payment_type' => strtoupper($payment->pay_type ?? 'N/A'),
                'transaction_id' => $payment->transaction_id ?? '-',
                'reference' => $payment->reference_number ?? '-',
                'paid_at' => $payment->payment_date ?? $payment->created_at,
                'created_at' => $payment->created_at,
            ];

            // Calcular totais
            if (strtolower($payment->status) === 'paid') {
                $totalPaid += $totalAmount;
            } else {
                $totalPending += $totalAmount;
            }

            $allMonthsData[] = $paymentData;
        }

        // Processar referências de pagamento (se não existir já em fee_assign)
        foreach ($paymentReferences as $payment) {
            $monthName = $payment->fee_month;

            // Verificar se já existe pagamento desse mês
            $exists = false;
            foreach ($allMonthsData as $existing) {
                if ($existing['month_name'] === $monthName) {
                    $exists = true;
                    break;
                }
            }

            if ($exists) {
                continue; // Pular se já existe pagamento desse mês
            }

            // Encontrar o número do mês
            $monthNumber = array_search($monthName, $monthNames) + 1;
            if ($monthNumber === false) {
                continue;
            }

            $paymentData = [
                'month_number' => $monthNumber,
                'month_name' => $monthName,
                'amount' => $payment->amount ?? 0,
                'base_amount' => $payment->amount ?? 0,
                'fine' => $payment->fine_amount ?? 0,
                'discount' => 0,
                'status' => $payment->status,
                'payment_type' => 'REFERÊNCIA',
                'transaction_id' => '-',
                'reference' => $payment->reference_number ?? '-',
                'paid_at' => $payment->paid_at,
                'created_at' => $payment->created_at,
            ];

            // Calcular totais conforme status
            if ($payment->status === 'paid') {
                $totalPaid += $payment->amount;
            } elseif ($payment->status === 'pending') {
                $totalPending += $payment->amount;
            } elseif ($payment->status === 'expired') {
                $totalExpired += $payment->amount;
            }

            $allMonthsData[] = $paymentData;
        }

        // Ordenar por número do mês
        usort($allMonthsData, function($a, $b) {
            return $a['month_number'] <=> $b['month_number'];
        });

        // Contar todos os status
        $countPaid = collect($allMonthsData)->where('status', 'paid')->count();
        $countExpired = collect($allMonthsData)->where('status', 'expired')->count();
        $countPending = collect($allMonthsData)->where('status', 'pending')->count();
        $countNotGenerated = 0; // Não mostramos mais "não gerados"

        // Determinar estado de pagamento
        if ($countExpired > 0) {
            $paymentStatus = 'IRREGULAR';
            $paymentStatusColor = '#dc3545'; // Vermelho
            $paymentStatusDetail = "({$countExpired} mensalidade(s) vencida(s)/atrasada(s))";
        } elseif ($countPending > 0) {
            $paymentStatus = 'PENDENTE';
            $paymentStatusColor = '#ffc107'; // Amarelo
            $paymentStatusDetail = "({$countPending} mensalidade(s) a pagar, não vencidas)";
        } else {
            $paymentStatus = 'EM DIA';
            $paymentStatusColor = '#28a745'; // Verde
            $paymentStatusDetail = "(Todos os pagamentos em dia)";
        }

        // Dados para o PDF
        $data = [
            'user' => $user,
            'classroom' => $classroom,
            'allMonthsData' => $allMonthsData,
            'totalPaid' => $totalPaid,
            'totalPending' => $totalPending,
            'totalExpired' => $totalExpired,
            'paymentStatus' => $paymentStatus,
            'paymentStatusColor' => $paymentStatusColor,
            'paymentStatusDetail' => $paymentStatusDetail,
            'countPaid' => $countPaid,
            'countExpired' => $countExpired,
            'countPending' => $countPending,
            'generatedAt' => now()->format('d/m/Y H:i'),
            'academicYear' => $currentYear,
        ];

        // Gerar PDF com otimizações de performance
        $pdf = Pdf::loadView('pdf.student-statement', $data)
            ->setOptions([
                'isRemoteEnabled' => false,
                'isHtml5ParserEnabled' => true,
                'isPhpEnabled' => false,
                'dpi' => 96, // Reduzir DPI para processar mais rápido
                'defaultFont' => 'DejaVu Sans'
            ]);

        // Nome do arquivo
        $filename = 'Extrato_' . $user->student_id . '_' . now()->format('Y-m-d') . '.pdf';

        // Download
        return $pdf->download($filename);
    }
}
