<?php

namespace App\Http\Livewire\Admin\Teachers;

use App\Models\Admin;
use App\Models\TeacherSubject;
use Livewire\Component;
use Livewire\WithPagination;
use LaravelMultipleGuards\Traits\FindGuard;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Note\Note;
class ListTeachers extends Component
{
    protected $lazy = false;
    use WithPagination, FindGuard, LivewireAlert;

    public $search, $delete;
    public $selectedClass = '';
    public $selectedTurma = '';
    public $selectedSubject = '';
    public $selectedStatus = '';
    public $turmas = [];
    public $subjects = [];

    protected $queryString = ['search', 'selectedClass', 'selectedTurma', 'selectedSubject', 'selectedStatus'];

    protected string $paginationTheme = 'bootstrap';

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function exportPDF()
    {
        return redirect()->route('admin.teachers.export.pdf', [
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'subject' => $this->selectedSubject,
            'status' => $this->selectedStatus,
            'search' => $this->search
        ]);
    }

    public function exportExcel()
    {
        return redirect()->route('admin.teachers.export.excel', [
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'subject' => $this->selectedSubject,
            'status' => $this->selectedStatus,
            'search' => $this->search
        ]);
    }

    public function updatedSelectedClass($value)
    {
        $this->selectedTurma = '';
        $this->selectedSubject = '';
        $this->resetPage();

        if ($value) {
            // Carregar turmas da classe selecionada
            $this->turmas = \App\Models\Classroom::where('class', $value)->get();

            // Carregar disciplinas que são ensinadas nessa classe
            $classroomIds = $this->turmas->pluck('id');
            $this->subjects = \App\Models\Subject::whereIn('id', function($query) use ($classroomIds) {
                $query->select('subject_id')
                    ->from('teacher_subjects')
                    ->whereIn('classroom_id', $classroomIds)
                    ->distinct();
            })->orderBy('name')->get();
        } else {
            $this->turmas = [];
            $this->subjects = [];
        }
    }

    public function updatedSelectedTurma()
    {
        $this->resetPage();
    }

    public function updatedSelectedSubject()
    {
        $this->resetPage();
    }

    public function updatedSelectedStatus()
    {
        $this->resetPage();
    }

    public bool $readyToLoad = false;

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function submit($id)
    {
        $this->delete = TeacherSubject::query()->firstWhere('id', $id);
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        //check if already enrolled
        // $check = TeacherSubject::query()->where('id', $this->delete->id)->get();
        // if (count($check) >= 1) {
        //     $this->alert('warning', 'sorry this model has subscribers.');
        //     return back();
        // }
        TeacherSubject::where('id', $this->delete->id)->delete();
        Note::createSystemNotification(Admin::class, 'Professor/Disciplina', 'Excluído com sucesso ');
        $this->emit('noteAdded');
        $this->alert('success', 'Professor/Disciplina - excluído com sucesso ');
        return back();


    }

    public function cancelled()
    {
        $this->alert('error', 'Cancelado.');
    }


    public function render()
    {
        // Buscar classes para os filtros
        $classes = \App\Models\Classroom::select('class')->distinct()->orderBy('class')->get();

        // Se nenhuma classe está selecionada, buscar todas as disciplinas
        // Se uma classe está selecionada, as disciplinas já foram carregadas em updatedSelectedClass
        if (!$this->selectedClass && empty($this->subjects)) {
            $this->subjects = \App\Models\Subject::orderBy('name')->get();
        }

        return view('livewire.admin.teachers.list-teachers', [
            'users' => $this->readyToLoad
                ? TeacherSubject::query()
                    ->latest('updated_at')
                    // Pesquisa geral
                    ->when($this->search, function ($query) {
                        $query->where(function ($subQuery) {
                            $subQuery->orWhere('teacher_id', 'like', '%' . $this->search . '%')
                                ->orWhere('classroom_id', 'like', '%' . $this->search . '%')
                                ->orWhereRelation('teacher', 'name', 'like', '%'. $this->search . '%')
                                ->orWhereRelation('teacher', 'email', 'like', '%'. $this->search . '%')
                                ->orWhereRelation('teacher', 'phone_number', 'like', '%'. $this->search . '%')
                                ->orWhere('class', 'like', '%' . $this->search . '%');
                        });
                    })
                    // Filtro por Classe
                    ->when($this->selectedClass, function ($query) {
                        $query->whereIn('classroom_id', function ($subQuery) {
                            $subQuery->select('id')
                                ->from('classrooms')
                                ->where('class', $this->selectedClass);
                        });
                    })
                    // Filtro por Turma específica
                    ->when($this->selectedTurma, function ($query) {
                        $query->where('classroom_id', $this->selectedTurma);
                    })
                    // Filtro por Disciplina
                    ->when($this->selectedSubject, function ($query) {
                        $query->where('subject_id', $this->selectedSubject);
                    })
                    // Filtro por Estado (Ativo/Suspenso)
                    ->when($this->selectedStatus !== '', function ($query) {
                        $query->whereHas('teacher', function ($subQuery) {
                            $subQuery->where('is_active', $this->selectedStatus);
                        });
                    })
                    ->paginate(10)
                : [],
            'classes' => $classes,
            'subjects' => $this->subjects
        ]);
    }
}
