<?php

namespace App\Http\Livewire\User\Fee;


use App\Models\Fee_assign;
use App\Models\User;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;

class FeeCollectionUser extends Component
{
    use FindGuard;
    public $studentId ;
    public $feeGroup;
    public $date;
    public $amount;
    public $discountGroup;
    public $discount = 0;
    public $fine = 0;
    public $paymentMode = 'Cash';
    public $note;
    public $students = [];
    public $selectedStudent;
    public $selectedFee;
    public $selectedFeeAssignId,$month, $year;


    protected $listeners = [
           'openModal' => 'preparePayment',
           'collectPayment' => 'processPayment'
       ];

       public function mount()
       {
           $user = $this->findGuardType()->user();
           if ($user) {
               $this->studentId = $user->id;
               $this->selectedStudent = User::findOrFail($this->studentId);
               $this->date = now()->format('d/m/Y');
           } else {
               abort(404, 'User not found');
           }
       }

       // Novo método para processar pagamentos via JavaScript
       public function processPayment($paymentData)
       {
           $this->fill([
               'amount' => $paymentData['amount'],
               'month' => $paymentData['month'],
               'year' => $paymentData['year'],
               'fine' => $paymentData['fine'] ?? 0,
               'discount' => $paymentData['discount'] ?? 0,
               'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
               'note' => $paymentData['note'] ?? null
           ]);

           $this->collectFee();
       }

       public function preparePayment($feeId, $amount, $month, $year)
       {
           $this->selectedFeeAssignId = $feeId;
           $this->amount = $amount;
           $this->month = $month;
           $this->year = $year;
           
           $this->dispatchBrowserEvent('openModal');
       }

       public function collectFee()
       {
           $this->validate([
               'amount' => 'required|numeric|min:0',
               'month' => 'required|string',
               'year' => 'required|numeric',
               'discount' => 'numeric|min:0',
               'fine' => 'numeric|min:0',
               'paymentMode' => 'required',
           ]);

           try {
               // Verificar se já existe pagamento para este mês/ano
               $existingPayment = Fee_assign::where([
                   'student_id' => $this->studentId,
                   'month' => $this->month,
                   'year' => $this->year
               ])->first();

               if ($existingPayment) {
                   throw new \Exception("Já existe um pagamento registrado para {$this->month}/{$this->year}");
               }

               Fee_assign::create([
                   'student_id' => $this->studentId,
                   'amount' => $this->amount,
                   'discount' => $this->discount,
                   'fine' => $this->fine,
                   'payment_mode' => $this->paymentMode,
                   'note' => $this->note,
                   'pay_day' => date("d"),
                   'pay_month' => date("m"),
                   'pay_year' => date("Y"),
                   'pay_type' => strtolower($this->paymentMode),
                   'status' => 'Paid',
                   'month' => $this->month,
                   'year' => $this->year,
               ]);

               $this->dispatchBrowserEvent('closeModal');
               $this->emit('paymentSuccess');
               
               session()->flash('message', 'Pagamento registrado com sucesso!');
               $this->reset(['amount', 'discount', 'fine', 'paymentMode', 'note', 'month', 'year']);

           } catch (\Exception $e) {
               $this->emit('paymentError', $e->getMessage());
               session()->flash('error', $e->getMessage());
           }
       }

    
    public function render()
    {
        $currentYear = date('Y');

        // Pegar aluno
        $student = $this->selectedStudent;
        $studentClassroom = $student->class->class ?? null;

        // Buscar fee structures ativas
        $feeStructures = FeeStructure::where('active', 1)->get();

        // Filtrar fee structures que aplicam à turma do aluno
        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
            $gradesArray = explode(',', $fee->grades);
            return in_array($studentClassroom, $gradesArray);
        });

        // Pegar fee assignments do aluno
        $fees = Fee_assign::where('student_id', $this->studentId)->get();

        // Inicializar totais
        $totalExpected = 0;
        $totalPaid = 0;
        $totalPending = 0;
        $totalOverdue = 0;

        // Lista de meses para exibir na tabela
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        $monthlyData = [];

        foreach ($months as $num => $name) {
            // Base fee mensal
            $baseFee = $filteredFeeStructures->sum('monthly_fee');

            // Data de vencimento: dia 5 de cada mês (podes mudar)
            $dueDate = Carbon::createFromDate($currentYear, $num, 5);
            $currentDate = Carbon::now();

            // Verificar multa se estiver atrasado
            $fine = 0;
            $latePenalty = $filteredFeeStructures->first()->late_penalty_percentage ?? 0;
            if ($currentDate->gt($dueDate)) {
                $fine = ($baseFee * $latePenalty) / 100;
            }

            // Procurar fee assignment pago para este mês
            $feeRecord = $fees->first(function ($f) use ($name, $currentYear) {
                return strtolower($f->month) == strtolower($name) && $f->year == $currentYear;
            });

            $paid = $feeRecord && strtolower($feeRecord->status) == 'paid';
            $amountPaid = $feeRecord->amount ?? 0;

            // Calcular totais
            $expected = $baseFee;
            $totalExpected += $expected;

            if ($paid) {
                $totalPaid += $amountPaid;
            } else {
                // Verificar se está em atraso
                if ($currentDate->gt($dueDate)) {
                    $totalOverdue += $expected + $fine;
                } else {
                    $totalPending += $expected;
                }
            }

            $monthlyData[] = [
                'month' => "$name $currentYear",
                'due_date' => $dueDate->format('d/m/Y'),
                'status' => $paid ? 'Pago' : ($currentDate->gt($dueDate) ? 'Em Atraso' : 'Não Pago'),
                'expected' => $expected,
                'fine' => $fine,
                'paid' => $amountPaid,
                'remaining' => max(0, $expected - $amountPaid),
            ];
        }

        return view('livewire.user.fee.fee-collection', [
            'student' => $student,
            'fees' => $fees,
            'feestructures' => $filteredFeeStructures,
            'monthlyData' => $monthlyData,
            'totalExpected' => $totalExpected,
            'totalPaid' => $totalPaid,
            'totalPending' => $totalPending,
            'totalOverdue' => $totalOverdue,
            'currentYear' => $currentYear,
        ]);
    }

    public function openModal($feeAssignId,$amount, $month, $year)
    {
        
        $this->selectedFeeAssignId = $feeAssignId; // Set the selected fee assign ID
        $this->month = (string) $month;  // Ensure month is a string
        $this->year = (int) $year;    
        $this->dispatchBrowserEvent('openModal'); // Dispatch event to open modal
        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
    }
}
