<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class FeeStructure extends Model
{
    use HasFactory;

    protected $table = 'fee_structures';

    protected $fillable = [
        'grade',
        'monthly_fee',
        'fee_name',
        'academic_year',
        'late_penalty_percentage',
        'payment_start_day',
        'payment_due_day',
        'active',
        'months',
        'grades',
        // use "penalty_type" como API correta; o alias abaixo cobre "palenty_type" na DB
        'penalty_type',
        'penalty_type', // mantém para compatibilidade se a coluna estiver assim na base
    ];

    protected $casts = [
        'grade' => 'integer',
        'monthly_fee' => 'decimal:2',
        'late_penalty_percentage' => 'decimal:2',
        'payment_start_day' => 'integer',
        'payment_due_day' => 'integer',
        'active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Escopo: apenas estruturas ativas
     */
    public function scopeActive($query)
    {
        return $query->where('active', 1);
    }

    /**
     * Texto amigável para status
     */
    public function getActiveStatusAttribute(): string
    {
        return $this->active ? 'Active' : 'Inactive';
    }

    /**
     * Relação com as atribuições de taxas
     */
    public function feeAssignments(): HasMany
    {
        return $this->hasMany(FeeAssign::class, 'fee_group_id');
    }

    /**
     * Escopo por classe (grade). Considera coluna "grades" CSV.
     */
    public function scopeForGrade($query, $grade)
    {
        return $query->where('grade', $grade)
                     ->orWhereRaw("FIND_IN_SET(?, grades)", [$grade]);
    }

    /**
     * Array de classes aplicáveis (a partir de "grades" CSV ou "grade" único)
     */
    public function getApplicableGradesAttribute(): array
    {
        if (!empty($this->grades)) {
            return array_values(array_filter(array_map('trim', explode(',', $this->grades))));
        }
        return $this->grade !== null ? [(int) $this->grade] : [];
    }

    /**
     * Array de meses aplicáveis (a partir de "months" CSV ou 1..12)
     */
    public function getApplicableMonthsAttribute(): array
    {
        if (!empty($this->months)) {
            return array_values(array_filter(array_map('intval', explode(',', $this->months))));
        }
        return range(1, 12);
    }

    /**
     * Alias para lidar com a coluna com typo "palenty_type" na DB.
     * Permite usar $model->penalty_type normalmente.
     */
    public function getPenaltyTypeAttribute()
    {
        // Prioriza "penalty_type" se existir; caso contrário, usa "palenty_type"
        return $this->attributes['penalty_type']
            ?? $this->attributes['penalty_type']
            ?? null;
    }

    public function setPenaltyTypeAttribute($value): void
    {
        // grava nas duas para manter compatibilidade enquanto migras
        $this->attributes['penalty_type'] = $value;
        $this->attributes['penalty_type'] = $value;
    }

    /**
     * Calcula a multa com base no tipo configurado.
     * Tipos suportados: percentage, fixed, daily
     */
    public function calculateFine(float $amount, int $daysLate = 0): float
    {
        if ($daysLate <= 0) {
            return 0.0;
        }

        $type = $this->penalty_type; // usa o alias acima
        $rate = (float) ($this->late_penalty_percentage ?? 0);

        switch ($type) {
            case 'percentage':
                return round($amount * ($rate / 100), 2);

            case 'fixed':
                // se usas valor fixo numa coluna separada, troca aqui.
                return round($rate, 2);

            case 'daily':
                return round($rate * $daysLate, 2);

            default:
                return 0.0;
        }
    }

    /**
     * Verifica se uma data está dentro do período de pagamento
     */
    public function isWithinPaymentPeriod($date = null): bool
    {
        $date = $date ?? now();
        $day = (int) $date->day;

        return $day >= (int) $this->payment_start_day
            && $day <= (int) $this->payment_due_day;
    }

    /**
     * Calcula dias de atraso com base na "payment_due_day"
     */
    public function calculateDaysLate($paymentDate = null): int
    {
        $paymentDate = $paymentDate ?? now();
        $due = (int) $this->payment_due_day;

        return max(0, (int) $paymentDate->day - $due);
    }
}

