<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ReceiptNumber extends Model
{
    use HasFactory;

    protected $table = 'receipt_numbers';

    protected $fillable = [
        'receipt_type',
        'record_id',
        'receipt_number',
        'issued_at',
        'issued_by',
        'user_id',
    ];

    protected $casts = [
        'issued_at' => 'datetime',
        'record_id' => 'integer',
        'user_id' => 'integer',
    ];

    /**
     * Relacionamento com o usuário que gerou o recibo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Scope para filtrar por tipo de recibo
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('receipt_type', $type);
    }

    /**
     * Scope para filtrar por ano
     */
    public function scopeForYear($query, $year)
    {
        return $query->whereYear('issued_at', $year);
    }

    /**
     * Scope para filtrar por quem emitiu
     */
    public function scopeIssuedBy($query, $issuedBy)
    {
        return $query->where('issued_by', $issuedBy);
    }

    /**
     * Obter registro original (Fee_assign ou PaymentReference)
     */
    public function getOriginalRecord()
    {
        if ($this->receipt_type === 'fee_assign') {
            return \App\Models\Fee_assign::find($this->record_id);
        } elseif ($this->receipt_type === 'payment_reference') {
            return \App\Models\PaymentReference::find($this->record_id);
        }
        return null;
    }
}
