<?php

namespace App\Http\Livewire\User\Account;

use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads; // Import the WithFileUploads trait
use Note\Note;

class UserProfile extends Component
{
    use FindGuard, LivewireAlert;
    use WithFileUploads; // Use the WithFileUploads trait

    public $user;
    public $name;
    public $phone_number;
    public $email;
    public $validatedData;
    public $avatar;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        $this->user = $this->findGuardType()->user();
        $this->name = $this->user->name;
        $this->phone_number = $this->user->phone_number;
        $this->email = $this->user->email;
        $this->avatar = $this->user->avatar; // Load existing avatar
    }

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'email', 'max:255'],
        'phone_number' => ['required', 'string'],
        'avatar' => ['nullable', 'image', 'max:2048'], // Maximum file size 2MB
    ];

    protected array $messages = [
        'email.email' => 'O formato do email é invalido!',
        'avatar.image' => 'O arquivo deve ser uma imagem.',
        'avatar.max' => 'O tamanho máximo da imagem é 2MB.',
    ];

    /**
     * Do real-time validations
     *
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
{
    $this->validatedData = $this->validate();

    // Check if avatar is uploaded
    if ($this->avatar) {
        // Get the original filename of the uploaded file
        $originalFilename = $this->avatar->getClientOriginalName();       
        // Define the folder where you want to store the avatars
        $folder = 'avatars';
        // Store the uploaded avatar in the specified folder using the original filename
        $avatarPath = $this->avatar->storeAs($folder, $originalFilename, 'avatar');        
        // Update the user's avatar path in the database
        $this->user->avatar = $avatarPath;      
        $this->user->save();
    }

    $this->confirm('Tem certeza de que deseja prosseguir?', [
        'toast' => false,
        'position' => 'center',
        'showConfirmButton' => true,
        'confirmButtonText' => 'Sim, eu tenho certeza!',
        'cancelButtonText' => 'Não, cancelar!',
        'onConfirmed' => 'confirmed',
        'onDismissed' => 'cancelled'
    ]);
}

    public function confirmed()
    {
        $this->user->fill($this->validatedData);

        if ($this->user->isClean()) {
            $this->alert('warning', 'Pelo menos um valor deve mudar.');
            return redirect()->back();
        }

        $this->user->save();
        Note::createSystemNotification(User::class, 'Perfil', 'Perfil atualizado com sucesso.');
        $this->emit('noteAdded');
        $this->alert('success', 'Perfil atualizado com sucesso.');
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view('livewire.user.account.user-profile');
    }
}

