<?php

namespace App\Helpers;

class MonthTranslator
{
    /**
     * Mapeamento de meses inglês -> português
     */
    private static $monthMap = [
        'January' => 'Janeiro',
        'February' => 'Fevereiro',
        'March' => 'Março',
        'April' => 'Abril',
        'May' => 'Maio',
        'June' => 'Junho',
        'July' => 'Julho',
        'August' => 'Agosto',
        'September' => 'Setembro',
        'October' => 'Outubro',
        'November' => 'Novembro',
        'December' => 'Dezembro'
    ];

    /**
     * Traduz um mês de inglês para português
     * Se já estiver em português, retorna como está
     */
    public static function toPortuguese(string $month): string
    {
        return self::$monthMap[$month] ?? $month;
    }

    /**
     * Traduz múltiplos meses (separados por vírgula)
     */
    public static function translateMultiple(string $months): string
    {
        $monthsArray = array_map('trim', explode(',', $months));
        $translated = array_map([self::class, 'toPortuguese'], $monthsArray);
        return implode(',', $translated);
    }

    /**
     * Traduz um mês de português para inglês
     */
    public static function toEnglish(string $month): string
    {
        $reverseMap = array_flip(self::$monthMap);
        return $reverseMap[$month] ?? $month;
    }

    /**
     * Verifica se o mês está em inglês
     */
    public static function isEnglish(string $month): bool
    {
        return array_key_exists($month, self::$monthMap);
    }

    /**
     * Verifica se o mês está em português
     */
    public static function isPortuguese(string $month): bool
    {
        return in_array($month, self::$monthMap);
    }

    /**
     * Normaliza para português (independente do formato de entrada)
     */
    public static function normalize(string $month): string
    {
        return self::toPortuguese($month);
    }
}
