<?php

namespace Chatify\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;
use App\Models\ChMessage as Message;
use App\Models\ChFavorite as Favorite;
use Chatify\Facades\ChatifyMessenger as Chatify;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Models\DeviceToken;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;


class MessagesController extends Controller
{
    protected $perPage = 30;

     /**
     * Authinticate the connection for pusher
     *
     * @param Request $request
     * @return void
     */
    public function pusherAuth(Request $request)
    {
        return Chatify::pusherAuth(
            $request->user(),
            Auth::user(),
            $request['channel_name'],
            $request['socket_id']
        );
    }

    /**
     * Fetch data by id for (user/group)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function idFetchData(Request $request)
    {
        return auth()->user();
        // Favorite
        $favorite = Chatify::inFavorite($request['id']);

        // User data
        if ($request['type'] == 'user') {
            $fetch = User::where('id', $request['id'])->first();
            if($fetch){
                $userAvatar = Chatify::getUserWithAvatar($fetch)->avatar;
            }
        }

        // send the response
        return Response::json([
            'favorite' => $favorite,
            'fetch' => $fetch ?? null,
            'user_avatar' => $userAvatar ?? null,
        ]);
    }

    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Illuminate\Http\JsonResponse
     */
    public function download($fileName)
    {
        $path = config('chatify.attachments.folder') . '/' . $fileName;
        if (Chatify::storage()->exists($path)) {
            return response()->json([
                'file_name' => $fileName,
                'download_path' => Chatify::storage()->url($path)
            ], 200);
        } else {
            return response()->json([
                'message'=>"Sorry, File does not exist in our server or may have been deleted!"
            ], 404);
        }
    }

    /**
     * Send a message to database
     *
     * @param Request $request
     * @return JSON response
     */
    public function send(Request $request)
    {
        // default variables
        $error = (object)[
            'status' => 0,
            'message' => null
        ];
        $attachment = null;
        $attachment_title = null;

        // if there is attachment [file]
        if ($request->hasFile('file')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();
            $allowed_files  = Chatify::getAllowedFiles();
            $allowed        = array_merge($allowed_images, $allowed_files);

            $file = $request->file('file');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed)) {
                    // get attachment name
                    $attachment_title = $file->getClientOriginalName();
                    // upload attachment and store the new name
                    $attachment = Str::uuid() . "." . $file->extension();
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
                } else {
                    $error->status = 1;
                    $error->message = "File extension not allowed!";
                }
            } else {
                $error->status = 1;
                $error->message = "File size you are trying to upload is too large!";
            }
        }

        if (!$error->status) {

             // send to database
             $message = Chatify::newMessage([
                'type' => $request['type'],
                'from_id' => Auth::user()->id,
                'to_channel_id' => $request['channel_id'],
                'body' => trim($request['message']),  // Removed htmlentities
                'attachment' => ($attachment) ? json_encode((object)[
                    'new_name' => $attachment,
                    'old_name' => trim($attachment_title),  // Removed htmlentities
                ]) : null,
                'reply_id' => $request->input('reply_id'),  // ✅ Save reply_id
            ]);

            // fetch message to send it with the response
            $messageData = Chatify::parseMessage($message);

            // send to user using pusher
            if (Auth::user()->id != $request['id']) {
                Chatify::push("private-chatify.".$request['id'], 'messaging', [
                    'from_id' => Auth::user()->id,
                    'to_id' => $request['id'],
                    'message' => $messageData
                ]);
            }

             // Send notification after successfully saving the message
             $this->sendNotification($request['channel_id'], $message, $attachment);


        }

        // send the response
        return Response::json([
            'status' => '200',
            'error' => $error,
            'message' => $messageData ?? [],
            'tempID' => $request['temporaryMsgId'],
        ]);
    }



    protected function sendNotification($channelId, $message, $attachment)
    {
        try {
            // Retrieve user IDs associated with the channel from the ch_channel_user table
            $userIds = DB::table('ch_channel_user')
                ->where('channel_id', $channelId)
                ->pluck('user_id');
    
            if ($userIds->isEmpty()) {
                throw new \Exception("No users found for channel_id: " . $channelId);
            }
    
            $firebaseCredentials = [
                "type" => "service_account",
                "project_id" => env('FIREBASE_PROJECT_ID'),
                "private_key_id" => env('FIREBASE_PRIVATE_KEY_ID'),
                "private_key" => str_replace("\\n", "\n", env('FIREBASE_PRIVATE_KEY')),
                "client_email" => env('FIREBASE_CLIENT_EMAIL'),
                "client_id" => env('FIREBASE_CLIENT_ID'),
                "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
                "token_uri" => "https://oauth2.googleapis.com/token",
                "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
                "client_x509_cert_url" => env('FIREBASE_CLIENT_X509_CERT_URL'),
                "universe_domain" => "googleapis.com"
            ];
    
            $factory = (new Factory)->withServiceAccount($firebaseCredentials);
            $messaging = $factory->createMessaging();
            Log::info('Firebase Messaging initialized');
    
            // For each user in the channel, send a notification
            foreach ($userIds as $userId) {
                // Don't send notification to the sender
                if ($userId == Auth::user()->id) {
                    continue;
                }

                $user = User::where('id', $userId)->first();
                if (!$user) {
                    Log::error("User not found for ID: $userId");
                    continue;
                }

                // Get device tokens for the user
                $tokens = DeviceToken::where('user_id', $user->id)->pluck('token');
                if ($tokens->isEmpty()) {
                    Log::error("No device tokens found for user: " . $user->name);
                    continue;
                }

                Log::info('Device tokens retrieved for user: ' . $user->name, ['tokens' => $tokens]);

                // Send notification for each device token
                foreach ($tokens as $token) {
                    try {
                        $notification = Notification::fromArray([
                            'title' => "Nova mensagem de " . Auth::user()->name,
                            'body' => ($attachment) ? "Ficheiro anexado" : $message->body,
                        ]);

                        $firebaseMessage = CloudMessage::withTarget('token', $token)
                            ->withNotification($notification)
                            ->withData([
                                'channel_id' => $channelId,
                                'sender_id' => Auth::user()->id,
                                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                                'type' => 'chat_message'
                            ])
                            ->withHighestPossiblePriority()
                            ->withDefaultSounds();

                        $messaging->send($firebaseMessage);
                        Log::info('Notification sent to token: ' . $token);

                    } catch (\Kreait\Firebase\Exception\Messaging\InvalidMessage $e) {
                        Log::error("Invalid token: {$token}. Error: {$e->getMessage()}");
                        DeviceToken::where('token', $token)->delete();  // Remove invalid token
                    } catch (\Exception $e) {
                        Log::error("Error sending notification: " . $e->getMessage());
                    }
                }
            }
    
        } catch (\Exception $e) {
            Log::error('Error sending notifications: ' . $e->getMessage());
        }
    }
    


    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JSON response
     */
    public function fetch(Request $request)
    {
        $query = Chatify::fetchMessagesQuery($request['id'])->latest();
        $messages = $query->paginate($request->per_page ?? $this->perPage);
        $totalMessages = $messages->total();
        $lastPage = $messages->lastPage();

        // ✅ Parse each message to include reply data
        $parsedMessages = collect($messages->items())->map(function($message) {
            return Chatify::parseMessage($message);
        })->toArray();

        $response = [
            'total' => $totalMessages,
            'last_page' => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages' => $parsedMessages,  // ✅ Return parsed messages with reply data
        ];
        return Response::json($response);
    }

    /**
     * Make messages as seen
     *
     * @param Request $request
     * @return void
     */
    public function seen(Request $request)
    {
        try {
            $userId = Auth::user()->id;
            // Aceitar tanto 'id' quanto 'channel_id' (JS envia 'channel_id')
            $channelId = $request['channel_id'] ?? $request['id'];

            Log::info('📩 [Chatify API] Marcando mensagens como vistas', [
                'user_id' => $userId,
                'channel_or_user_id' => $channelId,
                'request_data' => $request->all(),
            ]);

            // Tentar marcar como visto usando canal primeiro
            $updatedCount = Message::where('to_channel_id', $channelId)
                ->where('from_id', '!=', $userId) // Não marcar mensagens próprias
                ->where('seen', 0)
                ->update([
                    'seen' => 1,
                    'updated_at' => now()
                ]);

            // Se não atualizou nada, tentar método antigo (conversas diretas sem canal)
            if ($updatedCount === 0) {
                $updatedCount = Message::where('from_id', $channelId)
                    ->where('to_id', $userId)
                    ->where('seen', 0)
                    ->update([
                        'seen' => 1,
                        'updated_at' => now()
                    ]);
            }

            Log::info('✅ [Chatify API] Mensagens marcadas como vistas', [
                'channel_or_user_id' => $channelId,
                'updated_count' => $updatedCount,
            ]);

            return Response::json([
                'status' => 1,
                'updated_count' => $updatedCount,
            ], 200);

        } catch (\Exception $e) {
            Log::error('❌ [Chatify API] Erro ao marcar mensagens como vistas: ' . $e->getMessage());
            return Response::json([
                'status' => 0,
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get contacts list
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse response
     */
    public function getContacts(Request $request)
    {
        $users_list = [];
        // get all users that received/sent message from/to [Auth user]
        $users = Message::join('users',  function ($join) {
            $join->on('ch_messages.from_id', '=', 'users.id')
                ->orOn('ch_messages.to_id', '=', 'users.id');
        })
        ->where(function ($q) {
            $q->where('ch_messages.from_id', Auth::user()->id)
            ->orWhere('ch_messages.to_id', Auth::user()->id);
        })
        ->where('users.id','!=',Auth::user()->id)
        ->select('users.*',DB::raw('MAX(ch_messages.created_at) max_created_at'))
        ->orderBy('max_created_at', 'desc')
        ->groupBy('users.id')
        ->paginate($request->per_page ?? $this->perPage);

        return response()->json([
            'contacts' => $users->items(),
            'total' => $users->total() ?? 0,
            'last_page' => $users->lastPage() ?? 1,
        ], 200);
    }

    /**
     * Put a user in the favorites list
     *
     * @param Request $request
     * @return void
     */
    public function favorite(Request $request)
    {
        $userId = $request['user_id'];
        // check action [star/unstar]
        $favoriteStatus = Chatify::inFavorite($userId) ? 0 : 1;
        Chatify::makeInFavorite($userId, $favoriteStatus);

        // send the response
        return Response::json([
            'status' => @$favoriteStatus,
        ], 200);
    }

    /**
     * Get favorites list
     *
     * @param Request $request
     * @return void
     */
    public function getFavorites(Request $request)
    {
        $favorites = Favorite::where('user_id', Auth::user()->id)->get();
        foreach ($favorites as $favorite) {
            $favorite->user = User::where('id', $favorite->favorite_id)->first();
        }
        return Response::json([
            'total' => count($favorites),
            'favorites' => $favorites ?? [],
        ], 200);
    }

    /**
     * Search in messenger
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        $input = trim(filter_var($request['input']));
        $records = User::where('id','!=',Auth::user()->id)
                    ->where('name', 'LIKE', "%{$input}%")
                    ->paginate($request->per_page ?? $this->perPage);

        foreach ($records->items() as $index => $record) {
            $records[$index] += Chatify::getUserWithAvatar($record);
        }

        return Response::json([
            'records' => $records->items(),
            'total' => $records->total(),
            'last_page' => $records->lastPage()
        ], 200);
    }

    /**
     * Get shared photos
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function sharedPhotos(Request $request)
    {
        $images = Chatify::getSharedPhotos($request['user_id']);

        foreach ($images as $image) {
            $image = asset(config('chatify.attachments.folder') . $image);
        }
        // send the response
        return Response::json([
            'shared' => $images ?? [],
        ], 200);
    }

    /**
     * Delete conversation
     *
     * @param Request $request
     * @return void
     */
    public function deleteConversation(Request $request)
    {
        // delete
        $delete = Chatify::deleteConversation($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function updateSettings(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        // dark mode
        if ($request['dark_mode']) {
            $request['dark_mode'] == "dark"
                ? User::where('id', Auth::user()->id)->update(['dark_mode' => 1])  // Make Dark
                : User::where('id', Auth::user()->id)->update(['dark_mode' => 0]); // Make Light
        }

        // If messenger color selected
        if ($request['messengerColor']) {
            $messenger_color = trim(filter_var($request['messengerColor']));
            User::where('id', Auth::user()->id)
                ->update(['messenger_color' => $messenger_color]);
        }
        // if there is a [file]
        if ($request->hasFile('avatar')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();

            $file = $request->file('avatar');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    // delete the older one
                    if (Auth::user()->avatar != config('chatify.user_avatar.default')) {
                        $path = Chatify::getUserAvatarUrl(Auth::user()->avatar);
                        if (Chatify::storage()->exists($path)) {
                            Chatify::storage()->delete($path);
                        }
                    }
                    // upload
                    $avatar = Str::uuid() . "." . $file->extension();
                    $update = User::where('id', Auth::user()->id)->update(['avatar' => $avatar]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        // send the response
        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
        ], 200);
    }

    /**
     * Set user's active status
     *
     * @param Request $request
     * @return void
     */
    public function setActiveStatus(Request $request)
    {
        $activeStatus = $request['status'] > 0 ? 1 : 0;
        $status = User::where('id', Auth::user()->id)->update(['active_status' => $activeStatus]);
        return Response::json([
            'status' => $status,
        ], 200);
    }

    /**
     * Store typing status for a channel
     * When a user is typing (typing=1), store in cache with 5s TTL
     * When a user stops typing (typing=0), remove from cache
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function typing(Request $request)
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'channel_id' => 'required|uuid',
            'typing' => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            return Response::json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $channelId = $request->input('channel_id');
        $typing = $request->input('typing');
        $userId = Auth::id();

        // Cache key pattern: typing:{channel_id}:{user_id}
        $key = "typing:{$channelId}:{$userId}";

        if ($typing == 1) {
            // User is typing - store in cache with 5 seconds TTL
            Cache::put($key, now()->timestamp, 5);

            Log::info('💬 [Typing] User started typing', [
                'user_id' => $userId,
                'channel_id' => $channelId,
                'key' => $key
            ]);
        } else {
            // User stopped typing - remove from cache
            Cache::forget($key);

            Log::info('🛑 [Typing] User stopped typing', [
                'user_id' => $userId,
                'channel_id' => $channelId,
                'key' => $key
            ]);
        }

        return Response::json(['status' => 'success'], 200);
    }

    /**
     * Check if another user is typing in the channel
     * Returns typing status only for OTHER users (not the requester)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function typingStatus(Request $request)
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'channel_id' => 'required|uuid',
        ]);

        if ($validator->fails()) {
            return Response::json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $channelId = $request->input('channel_id');
        $currentUserId = Auth::id();

        // Pattern to search: typing:{channel_id}:*
        $pattern = "typing:{$channelId}:*";

        try {
            // Get all cache keys matching the pattern
            // Note: This works with Redis. For file/database cache, you might need a different approach
            $cacheStore = Cache::getStore();

            if (method_exists($cacheStore, 'getRedis')) {
                // Using Redis
                $redis = $cacheStore->getRedis();
                $keys = $redis->keys($pattern);

                foreach ($keys as $key) {
                    // Extract user_id from key: typing:{channel_id}:{user_id}
                    preg_match('/typing:[^:]+:(.+)$/', $key, $matches);
                    $typingUserId = $matches[1] ?? null;

                    // Ignore if it's the current user
                    if ($typingUserId && $typingUserId != $currentUserId) {
                        $timestamp = Cache::get($key);

                        // Check if still valid (less than 3 seconds old)
                        if ($timestamp && (now()->timestamp - $timestamp) < 3) {
                            // Optionally fetch user info
                            $user = User::find($typingUserId);

                            Log::info('✅ [Typing Status] Found active typing', [
                                'typing_user_id' => $typingUserId,
                                'typing_user_name' => $user->name ?? 'Unknown',
                                'channel_id' => $channelId,
                                'requester_id' => $currentUserId
                            ]);

                            return Response::json([
                                'is_typing' => true,
                                'user_id' => $typingUserId,
                                'user_name' => $user->name ?? null
                            ], 200);
                        }
                    }
                }
            } else {
                // Fallback for non-Redis cache drivers
                // Check if we can get the key directly
                // This is a simplified approach - ideally use Redis for this feature
                Log::warning('⚠️ [Typing Status] Non-Redis cache detected. Feature may not work optimally.');
            }

            return Response::json(['is_typing' => false], 200);

        } catch (\Exception $e) {
            Log::error('❌ [Typing Status] Error checking typing status: ' . $e->getMessage());

            // Return false on error to avoid breaking the app
            return Response::json(['is_typing' => false], 200);
        }
    }
}
