<?php

namespace App\Http\Livewire\Admin\Account;

use App\Models\Admin;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class AdminProfile extends Component
{
    use FindGuard, LivewireAlert;

    public $user;
    public $name;
    public $phone_number;
    public $email;
    public $validatedData;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        $this->user = $this->findGuardType()->user();
        $this->name = $this->user->name;
        $this->phone_number = $this->user->phone_number;
        $this->email = $this->user->email;
    }

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'email', 'max:255'],
        'phone_number' => ['required', 'string']
    ];

    protected array $messages = [
        'email.email' => 'O formato do endereço de e-mail não é válido.',
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $this->user->fill($this->validatedData);
        if ($this->user->isClean()) {
            $this->alert('warning', 'Pelo menos um valor deve mudar.');
            return redirect()->back();
        }

        $this->user->save();
        Note::createSystemNotification(Admin::class, 'Atualização do perfil', 'Seu perfil foi atualizado com sucesso.');
        $this->emit('noteAdded');
        $this->alert('success', 'Seu perfil foi atualizado com sucesso.');
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view('livewire.admin.account.admin-profile');
    }
}
