<?php

namespace App\Http\Livewire\Admin\Classes;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\User;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;
use Note\Note;

class ListClasses extends Component
{
    protected $lazy = false;
    use WithPagination, FindGuard, LivewireAlert;

    public $search, $delete;
    public $selectedClass = '';
    public $selectedStatus = '';

    protected $queryString = ['search', 'selectedClass', 'selectedStatus'];

    protected string $paginationTheme = 'bootstrap';

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedSelectedClass()
    {
        $this->resetPage();
    }

    public function updatedSelectedStatus()
    {
        $this->resetPage();
    }

    public function exportPDF()
    {
        return redirect()->route('admin.classes.export.pdf', [
            'search' => $this->search,
            'class' => $this->selectedClass,
            'status' => $this->selectedStatus
        ]);
    }

    public function exportExcel()
    {
        return redirect()->route('admin.classes.export.excel', [
            'search' => $this->search,
            'class' => $this->selectedClass,
            'status' => $this->selectedStatus
        ]);
    }

    public bool $readyToLoad = false;

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function submit($id)
    {
        $this->delete = Classroom::query()->firstWhere('id', $id);
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        //check if already enrolled
        $check = User::query()->where('classroom_id', $this->delete->id)->get();
        if (count($check) >= 1) {
            $this->alert('warning', 'Desculpe, este modelo tem subscritores.');
            return back();
        }
        Classroom::where('id', $this->delete->id)->delete();
        Note::createSystemNotification(Admin::class, 'Turma', 'Deletado com sucesso ');
        $this->emit('noteAdded');
        $this->alert('success', 'Deletado com sucesso ');
        return back();
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function toggleStudentStatus($classId, $status)
    {
        // Update the students' status based on class ID
        User::where('id', $classId)->update(['is_active' => $status]);
        Classroom::where('id', $classId)->update(['is_active' => $status]);
        // Alert for success
        $statusText = $status ? 'ativado' : 'desativado';
        $this->alert('success', "Todos os alunos foram {$statusText} com sucesso.");
    }

    public function render()
    {
        // Buscar classes distintas para o filtro
        $classes = Classroom::select('class')->distinct()->orderBy('class')->get();

        return view('livewire.admin.classes.list-classes', [
                'models' => $this->readyToLoad
                    ? Classroom::query()
                        ->when($this->search, function ($query) {
                            $query->where(function ($q) {
                                $q->where('name', 'like', '%' . $this->search . '%')
                                  ->orWhere('slug', 'like', '%' . $this->search . '%')
                                  ->orWhere('description', 'like', '%' . $this->search . '%');
                            });
                        })
                        // Filtro por Classe
                        ->when($this->selectedClass, function ($query) {
                            $query->where('class', $this->selectedClass);
                        })
                        // Filtro por Estado
                        ->when($this->selectedStatus !== '', function ($query) {
                            $query->where('is_active', $this->selectedStatus);
                        })
                        ->orderBy('class')
                        ->orderBy('name')
                        ->paginate(10)
                    : [],
                'classes' => $classes
            ]
        );
    }
}
