<?php

namespace App\Http\Livewire\Admin\Exams;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Exam;
use App\Models\Subject;
use App\Models\Trimester;
use App\Models\User;
use App\Models\TeacherSubject;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class BatchMarksEntry extends Component
{
    use FindGuard, LivewireAlert;

    public $classroom_id;
    public $subject_id;
    public $trimester_id;
    public $testType;
    public $year;
    public $class_level; // Added to track the class level (e.g., 5, 6, 7)

    public $students = [];
    public $marks = [];
    public $loadedStudents = false;

    // Variables to track permission issues
    public $permissionDenied = [];
    public $overridePermission = false;
    public $isTeacher = false;

    protected $listeners = [
        'confirmed',
        'cancelled',
        'overrideConfirmed',
        'overrideCancelled'
    ];

    // Define which mark fields should be treated as decimals
    protected $decimalMarkFields = ['ACS1a', 'ACS2a', 'AT', 'NE'];

    public function mount()
    {
        // Set default year to current year
        $this->year = date("Y");

        // Initialize for clean state
        $this->resetSelectionsExceptYear();
        $this->user = $this->findGuardType()->user();
        $this->isTeacher = ($this->user->role == 2);
    }

    /**
     * Reset all selections except the year
     */
    private function resetSelectionsExceptYear()
    {
        $this->class_level = null;
        $this->classroom_id = null;
        $this->subject_id = null;
        $this->trimester_id = null;
        $this->testType = null;
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
        $this->overridePermission = false;
    }

    /**
     * When year changes, reset all other selections
     */
    public function updatedYear()
    {
        $this->resetSelectionsExceptYear();
    }

    /**
     * When trimester changes, reset selections that depend on it
     */
    public function updatedTrimesterId()
    {
        $this->class_level = null;
        $this->classroom_id = null;
        $this->subject_id = null;
        $this->testType = null;
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
    }

    /**
     * When class level changes, reset classroom and subject
     */
    public function updatedClassLevel()
    {
        $this->classroom_id = null;
        $this->subject_id = null;
        $this->testType = null;
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
    }

    /**
     * When classroom changes, reset subject selection
     */
    public function updatedClassroomId()
    {
        $this->subject_id = null;
        $this->testType = null;
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
    }

    /**
     * When subject changes, reset test type
     */
    public function updatedSubjectId()
    {
        $this->testType = null;
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
    }

    /**
     * When test type changes, reset students
     */
    public function updatedTestType()
    {
        $this->students = [];
        $this->marks = [];
        $this->loadedStudents = false;
        $this->permissionDenied = [];
    }

    public function loadStudents()
    {
        $this->validate([
            'classroom_id' => 'required',
            'subject_id' => 'required',
            'trimester_id' => 'required',
            'testType' => 'required',
            'year' => 'required',
        ]);

        $this->students = User::where('classroom_id', $this->classroom_id)
            ->orderBy('name')
            ->get();

        // Initialize marks array with empty values
        foreach ($this->students as $student) {
            // First check if a record exists for this student, subject, trimester, and year (regardless of classroom)
            $existingExam = Exam::where('student_id', $student->id)
                ->where('subject_id', $this->subject_id)
                ->where('trimester_id', $this->trimester_id)
                ->where('year', $this->year)
                ->whereNull('deleted_at')
                ->first();

            // If no exact match found, check for classroom-specific match
            if (!$existingExam) {
                $existingExam = Exam::where('student_id', $student->id)
                    ->where('subject_id', $this->subject_id)
                    ->where('trimester_id', $this->trimester_id)
                    ->where('classroom_id', $this->classroom_id)
                    ->whereNull('deleted_at')
                    ->first();
            }

            // Set initial mark value based on existing record or empty
            if ($existingExam) {
                // Check if the current mark type already has a value > 0
                switch ($this->testType) {
                    case 1: // ACS1
                        $this->marks[$student->id] = $existingExam->ACS1a !== null ?
                                                    number_format((float)$existingExam->ACS1a, 2, '.', '') : '';
                        // Check if this would violate permission
                        if ($existingExam->ACS1a > 0) {
                            $this->permissionDenied[$student->id] = true;
                        }
                        break;
                    case 2: // ACS2
                        $this->marks[$student->id] = $existingExam->ACS2a !== null ?
                                                    number_format((float)$existingExam->ACS2a, 2, '.', '') : '';
                        // Check if this would violate permission
                        if ($existingExam->ACS2a > 0) {
                            $this->permissionDenied[$student->id] = true;
                        }
                        break;
                    case 3: // AT
                        $this->marks[$student->id] = $existingExam->AT !== null ?
                                                    number_format((float)$existingExam->AT, 2, '.', '') : '';
                        // Check if this would violate permission
                        if ($existingExam->AT > 0) {
                            $this->permissionDenied[$student->id] = true;
                        }
                        break;
                    case 4: // NE
                        $this->marks[$student->id] = $existingExam->NE !== null ?
                                                    number_format((float)$existingExam->NE, 2, '.', '') : '';
                        // Check if this would violate permission
                        if ($existingExam->NE > 0) {
                            $this->permissionDenied[$student->id] = true;
                        }
                        break;
                }
            } else {
                $this->marks[$student->id] = '';
                // No existing record, so no permission issues
                $this->permissionDenied[$student->id] = false;
            }
        }

        $this->loadedStudents = true;
    }

    /**
     * Format marks properly before validation and saving
     *
     * @param string $mark The mark to format
     * @return float|null The formatted mark
     */
    private function formatMark($mark)
    {
        if ($mark === '' || $mark === null) {
            return null;
        }

        // Replace comma with dot if present
        $mark = str_replace(',', '.', trim($mark));

        // Remove any trailing dots
        $mark = rtrim($mark, '.');

        return is_numeric($mark) ? (float) $mark : null;
    }

    public function saveMarks()
    {
        // Format marks and create validation rules
        $markRules = [];
        $formattedMarks = [];

        foreach ($this->marks as $studentId => $mark) {
            $formattedMark = $this->formatMark($mark);

            if ($formattedMark !== null) {
                $markRules["marks.$studentId"] = 'numeric|min:0|max:20';
                $formattedMarks[$studentId] = $formattedMark;
            } else {
                $formattedMarks[$studentId] = '';
            }
        }

        // Update marks with formatted values
        $this->marks = $formattedMarks;

        // Validate marks
        $this->validate($markRules);

        // Check if there are permission issues
        $hasPermissionIssues = !empty(array_filter($this->permissionDenied));

        if ($hasPermissionIssues && !$this->overridePermission) {
            // If there are permission issues, show a warning confirmation dialog
            $this->confirm('Algumas notas já possuem valores maiores que 0. Tem certeza de que deseja sobrescrever estas notas?', [
                'toast' => false,
                'position' => 'center',
                'showConfirmButton' => true,
                'confirmButtonText' => 'Sim, sobrescrever!',
                'cancelButtonText' => 'Não, manter originais!',
                'onConfirmed' => 'overrideConfirmed',
                'onDismissed' => 'overrideCancelled'
            ]);
            return;
        }

        // If no permission issues or override confirmed, proceed with normal confirmation
        $this->confirm('Tem certeza de que deseja salvar todas as notas?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar todas!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function overrideConfirmed()
    {
        // Set override permission to true to bypass permission checks
        $this->overridePermission = true;

        // Now call the normal confirmation process
        $this->confirmed();
    }

    public function overrideCancelled()
    {
        // User doesn't want to override existing values
        // Filter out marks for students with permission issues
        foreach ($this->permissionDenied as $studentId => $denied) {
            if ($denied) {
                $this->marks[$studentId] = '';
            }
        }

        // Then proceed with the save for other students
        $this->confirmed();
    }

    public function confirmed()
    {
        $updatedCount = 0;
        $createdCount = 0;
        $skippedCount = 0;
        $errorCount = 0;
        $errors = [];

        // Usar transação de banco de dados
        DB::beginTransaction();

        try {
            foreach ($this->marks as $studentId => $mark) {
                // Skip empty marks
                if ($mark === '') {
                    continue;
                }

                // Format mark to ensure it's stored correctly
                $formattedMark = $this->formatMark($mark);
                if ($formattedMark === null) {
                    $student = User::find($studentId);
                    $errors[] = [
                        'student_id' => $studentId,
                        'student_name' => $student ? $student->name : "ID: $studentId",
                        'error' => "Formato de nota inválido: '$mark'"
                    ];
                    $errorCount++;
                    continue;
                }

                // Validar intervalo
                if ($formattedMark < 0 || $formattedMark > 20) {
                    $student = User::find($studentId);
                    $errors[] = [
                        'student_id' => $studentId,
                        'student_name' => $student ? $student->name : "ID: $studentId",
                        'error' => "Nota fora do intervalo (0-20): $formattedMark"
                    ];
                    $errorCount++;
                    continue;
                }

                try {
                    // First check if a record exists for this student, subject, trimester, and year (regardless of classroom)
                    $existingExam = Exam::where('student_id', $studentId)
                    ->where('subject_id', $this->subject_id)
                    ->where('trimester_id', $this->trimester_id)
                    ->where('classroom_id', $this->classroom_id)
                    ->where('year', $this->year)
                    ->first();

                if ($existingExam) {
                    // If the record exists but has a different classroom, update the classroom
                    if ($existingExam->classroom_id != $this->classroom_id) {
                        $existingExam->classroom_id = $this->classroom_id;
                    }

                    // Update the appropriate field based on test type
                    $fieldUpdated = false;
                    switch ($this->testType) {
                        case 1: // ACS1
                            // Check if we have permission to update (value is 0 or override is enabled)
                            if ($existingExam->ACS1a <= 0 || $this->overridePermission) {
                                $existingExam->ACS1a = $formattedMark;
                                $fieldUpdated = true;
                            } else {
                                $skippedCount++;
                            }
                            break;
                        case 2: // ACS2
                            if ($existingExam->ACS2a <= 0 || $this->overridePermission) {
                                $existingExam->ACS2a = $formattedMark;
                                $fieldUpdated = true;
                            } else {
                                $skippedCount++;
                            }
                            break;
                        case 3: // AT
                            if ($existingExam->AT <= 0 || $this->overridePermission) {
                                $existingExam->AT = $formattedMark;
                                $fieldUpdated = true;
                            } else {
                                $skippedCount++;
                            }
                            break;
                        case 4: // NE
                            if ($existingExam->NE <= 0 || $this->overridePermission) {
                                $existingExam->NE = $formattedMark;
                                $fieldUpdated = true;
                            } else {
                                $skippedCount++;
                            }
                            break;
                    }

                    if ($fieldUpdated) {
                        // Ensure year is set
                        $existingExam->year = $this->year;
                        $existingExam->save();
                        $updatedCount++;
                    }
                } else {
                    // Create a new exam record
                    $newExam = new Exam();
                    $newExam->student_id = $studentId;
                    $newExam->subject_id = $this->subject_id;
                    $newExam->trimester_id = $this->trimester_id;
                    $newExam->classroom_id = $this->classroom_id;
                    $newExam->year = $this->year;

                    // Set the appropriate field based on test type
                    switch ($this->testType) {
                        case 1: // ACS1
                            $newExam->ACS1a = $formattedMark;
                            break;
                        case 2: // ACS2
                            $newExam->ACS2a = $formattedMark;
                            break;
                        case 3: // AT
                            $newExam->AT = $formattedMark;
                            break;
                        case 4: // NE
                            $newExam->NE = $formattedMark;
                            break;
                    }

                    $newExam->save();
                    $createdCount++;
                }
                } catch (\Exception $e) {
                    $student = User::find($studentId);
                    $errors[] = [
                        'student_id' => $studentId,
                        'student_name' => $student ? $student->name : "ID: $studentId",
                        'error' => "Erro ao salvar: " . $e->getMessage()
                    ];
                    $errorCount++;
                    // Log the error for debugging
                    \Log::error("Error saving exam mark for student {$studentId}: " . $e->getMessage());
                }
            }

            // Se não houve erros críticos, fazer commit
            DB::commit();

            // Create notification
            $message = "Notas atualizadas para {$updatedCount} estudantes, criadas para {$createdCount} estudantes";
            if ($skippedCount > 0) {
                $message .= ", {$skippedCount} notas não alteradas (valor > 0)";
            }
            if ($errorCount > 0) {
                $message .= ", {$errorCount} erros encontrados";
            }

            Note::createSystemNotification(Admin::class, 'Notas', $message);
            $this->emit('noteAdded');

            // Se houver erros, mostrar lista detalhada
            if ($errorCount > 0 && !empty($errors)) {
                $errorList = "<div style='max-height: 300px; overflow-y: auto;'><ul class='mb-0'>";
                foreach ($errors as $error) {
                    $errorList .= "<li><strong>{$error['student_name']}</strong>: {$error['error']}</li>";
                }
                $errorList .= "</ul></div>";

                $this->alert('warning', "Notas parcialmente salvas. {$updatedCount} atualizadas, {$createdCount} criadas, {$skippedCount} não alteradas, {$errorCount} erros:<br>" . $errorList, [
                    'html' => true
                ]);
            } else if ($skippedCount > 0) {
                $this->alert('info', "Notas salvas com sucesso! ({$updatedCount} atualizadas, {$createdCount} criadas, {$skippedCount} não alteradas por já terem valores maiores que 0)");
            } else {
                $this->alert('success', "Notas salvas com sucesso! ({$updatedCount} atualizadas, {$createdCount} criadas)");
            }

            // Reset the override permission flag
            $this->overridePermission = false;

            // Reload students to refresh the marks
            $this->loadStudents();

        } catch (\Exception $e) {
            // Se houver erro crítico, fazer rollback
            DB::rollback();

            \Log::error('Critical error in batch marks save: ' . $e->getMessage());

            $this->alert('error', 'Erro crítico ao salvar notas. Todas as alterações foram revertidas. Por favor, tente novamente ou contacte o suporte.');

            $this->overridePermission = false;
        }
    }

    public function cancelled()
    {
        $this->alert('error', 'Operação cancelada.');
        $this->overridePermission = false;
    }

    public function render()
    {
        $user = $this->findGuardType()->user();
        $this->isTeacher = ($user->role == 2);

        // Get available class levels for the selected year
        $availableClassLevels = [];

        if ($this->isTeacher) {
            // For teachers, get class levels based on their assignments and exams
            if (!empty($this->year) && !empty($this->trimester_id)) {
                // Get class levels from exams for this year and trimester
                $examClassLevels = DB::table('exams')
                    ->join('teacher_subjects', function ($join) {
                        $join->on('exams.subject_id', '=', 'teacher_subjects.subject_id');
                    })
                    ->join('classrooms', 'exams.classroom_id', '=', 'classrooms.id')
                    ->where('teacher_subjects.teacher_id', '=', $user->id)
                    ->where('exams.year', '=', $this->year)
                    ->where('exams.trimester_id', '=', $this->trimester_id)
                    ->select('classrooms.class')
                    ->distinct()
                    ->pluck('class')
                    ->toArray();

                // Also get class levels directly assigned to this teacher
                $directClassLevels = TeacherSubject::where('teacher_id', $user->id)
                    ->whereNotNull('class')
                    ->distinct()
                    ->pluck('class')
                    ->toArray();

                //$availableClassLevels = array_values(array_unique(array_merge($examClassLevels, $directClassLevels)));
                $availableClassLevels = TeacherSubject::where('teacher_id', $user->id)
                    ->whereNotNull('class')
                    ->distinct()
                    ->pluck('class')
                    ->sort()
                    ->values()
                    ->toArray();
            }
        }

        // Get classrooms based on selected class level
        $classrooms = collect();
        if (!empty($this->class_level)) {
            if ($this->isTeacher) {
                // For teachers, show only classrooms they are assigned to
                $classrooms = Classroom::where('class', $this->class_level)
                    ->where(function($query) use ($user) {
                        // Classrooms directly assigned to this teacher
                        $query->whereIn('id', function($subquery) use ($user) {
                            $subquery->select('classroom_id')
                                    ->from('teacher_subjects')
                                    ->where('teacher_id', $user->id)
                                    ->whereNotNull('classroom_id');
                        });

//                        // OR Classrooms where teacher has exams for this year and trimester
//                        if (!empty($this->year) && !empty($this->trimester_id)) {
//                            $query->orWhereIn('id', function($subquery) use ($user) {
//                                $subquery->select('exams.classroom_id')
//                                        ->from('exams')
//                                        ->join('teacher_subjects', 'exams.subject_id', '=', 'teacher_subjects.subject_id')
//                                        ->where('teacher_subjects.teacher_id', '=', $user->id)
//                                        ->where('exams.year', '=', $this->year)
//                                        ->where('exams.trimester_id', '=', $this->trimester_id)
//                                        ->distinct();
//                            });
//                        }
                    })
                    ->orderBy('name')
                    ->get();
            } else {
                // For admins, show all classrooms for the selected class level
                $classrooms = Classroom::where('class', $this->class_level)
                    ->orderBy('name')
                    ->get();
            }
        }

        // Get subjects based on selected classroom
        $subjects = collect();
        if (!empty($this->classroom_id)) {
            $selectedClassroom = Classroom::find($this->classroom_id);

            if ($selectedClassroom) {
                if ($this->isTeacher) {
                    // For teachers, show only subjects they teach in this classroom
                    $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                        ->select('subjects.id', 'subjects.name', 'subjects.slug')
                        ->where('teacher_subjects.teacher_id', $user->id)
                        ->where(function($query) use ($selectedClassroom) {
                            $query->where('teacher_subjects.classroom_id', $selectedClassroom->id)
                                 ->orWhere(function($q) use ($selectedClassroom) {
                                     $q->where('teacher_subjects.class', $selectedClassroom->class)
                                       ->whereNull('teacher_subjects.classroom_id');
                                 });
                        })
                        ->orderBy('subjects.name')
                        ->distinct()
                        ->get();

                    // If the teacher has exams for this classroom, also include those subjects
                    if (!empty($this->year) && !empty($this->trimester_id)) {
                        $examSubjects = Subject::join('exams', 'subjects.id', '=', 'exams.subject_id')
                            ->join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                            ->select('subjects.id', 'subjects.name', 'subjects.slug')
                            ->where('teacher_subjects.teacher_id', $user->id)
                            ->where('exams.classroom_id', $this->classroom_id)
                            ->where('exams.year', $this->year)
                            ->where('exams.trimester_id', $this->trimester_id)
                            ->orderBy('subjects.name')
                            ->distinct()
                            ->get();

                        // Merge both lists and remove duplicates
                        $subjects = $subjects->merge($examSubjects)->unique('id')->values();
                    }
                } else {
                    // For admins, show all subjects for this classroom or class level
                    $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                        ->select('subjects.id', 'subjects.name', 'subjects.slug')
                        ->where(function($query) use ($selectedClassroom) {
                            $query->where('teacher_subjects.classroom_id', $selectedClassroom->id)
                                 ->orWhere(function($q) use ($selectedClassroom) {
                                     $q->where('teacher_subjects.class', $selectedClassroom->class)
                                       ->whereNull('teacher_subjects.classroom_id');
                                 });
                        })
                        ->orderBy('subjects.name')
                        ->distinct()
                        ->get();
                }
            }
        }

        return view('livewire.admin.exams.batch-marks-entry', [
            'availableClassLevels' => $availableClassLevels,
            'classrooms' => $classrooms,
            'subjects' => $subjects,
            'trimesters' => Trimester::orderBy('name')->get(),
        ]);
    }
}
