<?php

namespace App\Http\Livewire\Admin\Reports;

use App\Models\Classroom;
use App\Models\Fee_assign;
use App\Models\User;
use App\Models\PaymentReference;
use App\Models\ReceiptNumber;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;

class PaymentReports extends Component
{
    use WithPagination;

    // Filtros
    public $filter_type = 'class'; // class, reference, date, student, all, receipt
    public $class_id = '';
    public $reference_number = '';
    public $receipt_number = ''; // Receipt Number
    public $date_from = '';
    public $date_to = '';
    public $student_id = '';
    public $payment_status = ''; // all, paid, pending
    public $search_student = '';
    public $transaction_id = ''; // POS Transaction ID

    // Para filtro de estudante hierárquico (Classe → Turma → Estudante)
    public $student_class_name = ''; // Nome da classe (ex: 10, 11, 12)
    public $student_classroom_id = ''; // ID da turma específica

    // Resultados
    public $showResults = false;
    public $totalAmount = 0;
    public $totalCount = 0;
    public $totalFine = 0;
    public $subtotal = 0;
    public $students = [];

    protected $queryString = [
        'filter_type',
        'class_id',
        'reference_number',
        'receipt_number',
        'transaction_id',
        'date_from',
        'date_to',
        'student_id',
        'payment_status'
    ];

    public function mount()
    {
        $this->date_from = now()->startOfMonth()->format('Y-m-d');
        $this->date_to = now()->format('Y-m-d');
    }

    public function updatingFilterType()
    {
        $this->resetFilters();
    }

    public function updatingClassId()
    {
        $this->resetPage();
    }

    public function updatingStudentClassName()
    {
        // Quando muda a classe, reseta turma e estudante
        $this->student_classroom_id = '';
        $this->student_id = '';
    }

    public function updatingStudentClassroomId()
    {
        // Quando muda a turma, reseta estudante
        $this->student_id = '';
    }

    public function resetFilters()
    {
        $this->class_id = '';
        $this->reference_number = '';
        $this->receipt_number = '';
        $this->transaction_id = '';
        $this->student_id = '';
        $this->search_student = '';
        $this->student_class_name = '';
        $this->student_classroom_id = '';
        $this->showResults = false;
        $this->resetPage();
    }

    public function generateReport()
    {
        $this->validate($this->getValidationRules());
        $this->showResults = true;
        $this->resetPage();
    }

    private function getValidationRules()
    {
        $rules = [
            'date_from' => 'required|date',
            'date_to' => 'required|date|after_or_equal:date_from',
        ];

        if ($this->filter_type === 'class') {
            $rules['class_id'] = 'required|exists:classrooms,id';
        } elseif ($this->filter_type === 'reference') {
            $rules['reference_number'] = 'required|string|size:11';
        } elseif ($this->filter_type === 'receipt') {
            $rules['receipt_number'] = 'required|string';
        } elseif ($this->filter_type === 'transaction') {
            $rules['transaction_id'] = 'required|string';
        } elseif ($this->filter_type === 'student') {
            $rules['student_id'] = 'required|exists:users,id';
        }

        return $rules;
    }

    public function getResultsProperty()
    {
        if (!$this->showResults) {
            return collect([]);
        }

        $query = Fee_assign::query()
            ->with(['student', 'student.classroom', 'paymentReference'])
            ->whereBetween('created_at', [
                Carbon::parse($this->date_from)->startOfDay(),
                Carbon::parse($this->date_to)->endOfDay()
            ]);

        // Aplicar filtros
        if ($this->filter_type === 'class' && $this->class_id) {
            $query->whereHas('student', function ($q) {
                $q->where('classroom_id', $this->class_id);
            });
        } elseif ($this->filter_type === 'reference' && $this->reference_number) {
            // Buscar pagamentos através do relacionamento com payment_references
            $query->whereHas('paymentReference', function ($q) {
                $q->where('reference_number', $this->reference_number);
            });
        } elseif ($this->filter_type === 'receipt' && $this->receipt_number) {
            // Buscar por número de recibo
            $receiptIds = ReceiptNumber::where('receipt_number', $this->receipt_number)
                ->where('receipt_type', 'fee_assign')
                ->pluck('record_id');
            $query->whereIn('id', $receiptIds);
        } elseif ($this->filter_type === 'transaction' && $this->transaction_id) {
            // Buscar por POS Transaction ID
            $query->where('transaction_id', $this->transaction_id);
        } elseif ($this->filter_type === 'student' && $this->student_id) {
            $query->where('student_id', $this->student_id);
        }

        // Status de pagamento
        if ($this->payment_status === 'paid') {
            $query->whereNotNull('amount')->where('amount', '>', 0);
        } elseif ($this->payment_status === 'pending') {
            $query->where(function ($q) {
                $q->whereNull('amount')->orWhere('amount', '=', 0);
            });
        }

        $results = $query->latest('created_at')->paginate(50);

        // Calcular totais (sem paginação)
        // Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
        $totalsQuery = clone $query;
        $allPayments = $totalsQuery->get();

        $this->subtotal = $allPayments->sum('amount') ?? 0; // Valor base (sem multa)
        $this->totalFine = $allPayments->sum('fine') ?? 0; // Multa total
        $totalDiscount = $allPayments->sum('discount') ?? 0; // Desconto total
        $this->totalAmount = $this->subtotal + $this->totalFine - $totalDiscount; // Total final
        $this->totalCount = $allPayments->count() ?? 0;

        return $results;
    }

    public function exportPDF()
    {
        $this->validate($this->getValidationRules());

        $results = $this->getReportData();

        // FORÇAR DEBUG - Escrever em arquivo separado
        file_put_contents(storage_path('logs/pdf_debug.txt'),
            "ExportPDF chamado em: " . now() . "\n" .
            "subtotal: " . $this->subtotal . "\n" .
            "totalFine: " . $this->totalFine . "\n" .
            "totalAmount: " . $this->totalAmount . "\n" .
            "totalCount: " . $this->totalCount . "\n" .
            "results_count: " . $results->count() . "\n"
        );

        $pdf = Pdf::loadView('livewire.admin.reports.payment-reports-pdf', [
            'results' => $results,
            'filter_type' => $this->filter_type,
            'filters' => $this->getFilterSummary(),
            'total_amount' => $this->totalAmount,
            'total_count' => $this->totalCount,
            'total_fine' => $this->totalFine,
            'subtotal' => $this->subtotal,
            'date_from' => $this->date_from,
            'date_to' => $this->date_to,
        ])->setPaper('a4', 'landscape');

        return response()->streamDownload(function () use ($pdf) {
            echo $pdf->output();
        }, 'relatorio-pagamentos-' . now()->format('Y-m-d-His') . '.pdf');
    }

    public function exportExcel()
    {
        $this->validate($this->getValidationRules());

        return Excel::download(
            new \App\Exports\PaymentReportsExport(
                $this->getReportData(),
                $this->getFilterSummary(),
                $this->totalAmount,
                $this->totalCount,
                $this->totalFine,
                $this->subtotal
            ),
            'relatorio-pagamentos-' . now()->format('Y-m-d-His') . '.xlsx'
        );
    }

    private function getReportData()
    {
        $query = Fee_assign::query()
            ->with(['student', 'student.classroom', 'paymentReference'])
            ->whereBetween('created_at', [
                Carbon::parse($this->date_from)->startOfDay(),
                Carbon::parse($this->date_to)->endOfDay()
            ]);

        // Aplicar mesmos filtros
        if ($this->filter_type === 'class' && $this->class_id) {
            $query->whereHas('student', function ($q) {
                $q->where('classroom_id', $this->class_id);
            });
        } elseif ($this->filter_type === 'reference' && $this->reference_number) {
            // Buscar pagamentos através do relacionamento com payment_references
            $query->whereHas('paymentReference', function ($q) {
                $q->where('reference_number', $this->reference_number);
            });
        } elseif ($this->filter_type === 'receipt' && $this->receipt_number) {
            // Buscar por número de recibo
            $receiptIds = ReceiptNumber::where('receipt_number', $this->receipt_number)
                ->where('receipt_type', 'fee_assign')
                ->pluck('record_id');
            $query->whereIn('id', $receiptIds);
        } elseif ($this->filter_type === 'transaction' && $this->transaction_id) {
            // Buscar por POS Transaction ID
            $query->where('transaction_id', $this->transaction_id);
        } elseif ($this->filter_type === 'student' && $this->student_id) {
            $query->where('student_id', $this->student_id);
        }

        if ($this->payment_status === 'paid') {
            $query->whereNotNull('amount')->where('amount', '>', 0);
        } elseif ($this->payment_status === 'pending') {
            $query->where(function ($q) {
                $q->whereNull('amount')->orWhere('amount', '=', 0);
            });
        }

        // Recalcular totais
        // Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
        $totalsQuery = clone $query;
        $allPayments = $totalsQuery->get();

        $this->subtotal = $allPayments->sum('amount') ?? 0; // Valor base (sem multa)
        $this->totalFine = $allPayments->sum('fine') ?? 0; // Multa total
        $totalDiscount = $allPayments->sum('discount') ?? 0; // Desconto total
        $this->totalAmount = $this->subtotal + $this->totalFine - $totalDiscount; // Total final
        $this->totalCount = $allPayments->count() ?? 0;

        // DEBUG: Log dos cálculos
        \Log::info('GetReportData - Cálculos:', [
            'count' => $this->totalCount,
            'sum_amount' => $allPayments->sum('amount'),
            'sum_fine' => $allPayments->sum('fine'),
            'sum_discount' => $totalDiscount,
            'subtotal_calculado' => $this->subtotal,
            'totalFine_calculado' => $this->totalFine,
            'totalAmount_calculado' => $this->totalAmount
        ]);

        return $allPayments;
    }

    private function getFilterSummary()
    {
        $summary = [];

        if ($this->filter_type === 'class' && $this->class_id) {
            $classroom = Classroom::find($this->class_id);
            $summary[] = 'Turma: ' . ($classroom ? $classroom->class . ' - ' . $classroom->name : 'N/A');
        } elseif ($this->filter_type === 'reference' && $this->reference_number) {
            $summary[] = 'Referência: ' . $this->reference_number;
        } elseif ($this->filter_type === 'receipt' && $this->receipt_number) {
            $summary[] = 'Número de Recibo: ' . $this->receipt_number;
        } elseif ($this->filter_type === 'student' && $this->student_id) {
            $student = User::find($this->student_id);
            $summary[] = 'Estudante: ' . ($student ? $student->name : 'N/A');
        } else {
            $summary[] = 'Todos os pagamentos';
        }

        $summary[] = 'Período: ' . Carbon::parse($this->date_from)->format('d/m/Y') . ' a ' . Carbon::parse($this->date_to)->format('d/m/Y');

        if ($this->payment_status) {
            $summary[] = 'Status: ' . ($this->payment_status === 'paid' ? 'Pagos' : 'Pendentes');
        }

        return $summary;
    }

    public function getClassroomsProperty()
    {
        return Classroom::orderBy('class')->orderBy('name')->get();
    }

    // Buscar nomes de classes únicas (10, 11, 12, etc)
    public function getAvailableClassesProperty()
    {
        return Classroom::select('class')
            ->distinct()
            ->orderBy('class')
            ->pluck('class');
    }

    // Buscar turmas de uma classe específica
    public function getClassroomsByClassProperty()
    {
        if (!$this->student_class_name) {
            return collect([]);
        }

        return Classroom::where('class', $this->student_class_name)
            ->orderBy('name')
            ->get();
    }

    // Buscar estudantes de uma turma específica
    public function getStudentsByClassroomProperty()
    {
        if (!$this->student_classroom_id) {
            return collect([]);
        }

        return User::where('classroom_id', $this->student_classroom_id)
            ->orderBy('name')
            ->get();
    }

    public function getStudentsForSelectProperty()
    {
        // Este método será usado apenas se não houver filtro hierárquico
        if ($this->student_classroom_id) {
            return $this->students_by_classroom;
        }

        $query = User::query()
            ->orderBy('name');

        if ($this->search_student) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->search_student . '%')
                  ->orWhere('email', 'like', '%' . $this->search_student . '%');
            });
        }

        if ($this->class_id && $this->filter_type === 'student') {
            $query->where('classroom_id', $this->class_id);
        }

        return $query->limit(50)->get();
    }

    public function render()
    {
        return view('livewire.admin.reports.payment-reports', [
            'results' => $this->results,
            'classrooms' => $this->classrooms,
            'students_for_select' => $this->students_for_select,
            'available_classes' => $this->available_classes,
            'classrooms_by_class' => $this->classrooms_by_class,
            'students_by_classroom' => $this->students_by_classroom,
        ]);
    }
}
