<?php

namespace App\Http\Livewire\User\Fee;

use App\Models\Fee_assign;
use App\Models\User;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;

class FeeCollectionUser extends Component
{
    use FindGuard;

    public $studentId;
    public $amount, $discount = 0, $fine = 0;
    public $paymentMode = 'Cash', $note;
    public $selectedStudent;
    public $selectedFeeAssignId, $month, $year;

    // Totais
    public $totalBase = 0, $totalPaid = 0, $pending = 0, $withFine = 0, $percentPaid = 0, $dueDate = 'N/A';

    public function mount()
    {
        $user = $this->findGuardType()->user();
        if (!$user) abort(404, 'User not found');

        $this->studentId = $user->id;
        $this->selectedStudent = User::findOrFail($this->studentId);

        $this->calcularTotais();
    }

    public function calcularTotais()
    {
        $studentClassroom = $this->selectedStudent->class->class ?? null;

        $feeStructures = FeeStructure::where('active', 1)->get();
        $filtered = $feeStructures->filter(function ($feeStructure) use ($studentClassroom) {
            return in_array($studentClassroom, explode(',', $feeStructure->grades));
        });

        $this->totalBase = $filtered->sum('monthly_fee');

        // Busca vencimento da primeira estrutura
        $structure = $filtered->first();
        if ($structure && $structure->payment_due_day) {
            $this->dueDate = Carbon::parse($structure->payment_due_day)->format('d/m/Y');
        }

        // Pagamentos
        $fees = Fee_assign::where('student_id', $this->studentId)->get();
        $this->totalPaid = $fees->where('status', 'Paid')->sum('amount');
        $this->pending = max($this->totalBase - $this->totalPaid, 0);
        $this->percentPaid = $this->totalBase > 0 ? round(($this->totalPaid / $this->totalBase) * 100, 2) : 0;

        // Calcula multa se houver atraso
        if ($structure && $structure->late_penalty_percentage && Carbon::now()->gt(Carbon::parse($structure->payment_due_day))) {
            $this->withFine = $this->pending + ($this->pending * $structure->late_penalty_percentage / 100);
        } else {
            $this->withFine = $this->pending;
        }
    }

    public function collectFee()
    {
        $this->validate([
            'amount' => 'required|numeric|min:0',
            'discount' => 'numeric|min:0',
            'fine' => 'numeric|min:0',
            'paymentMode' => 'required',
        ]);

        Fee_assign::create([
            'student_id' => $this->studentId,
            'amount' => $this->amount,
            'discount' => $this->discount,
            'fine' => $this->fine,
            'payment_mode' => $this->paymentMode,
            'note' => $this->note,
            'pay_day' => date("d"),
            'pay_month' => date("m"),
            'pay_year' => date("Y"),
            'pay_type' => 'cash',
            'status' => 'Paid',
            'month' => $this->month,
            'year' => $this->year,
        ]);

        session()->flash('message', 'Pagamento efetuado com sucesso.');

        $this->reset(['amount', 'discount', 'fine', 'paymentMode', 'note']);
        $this->dispatchBrowserEvent('closeModal');

        $this->calcularTotais();
    }

    public function openModal($feeAssignId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeAssignId;
        $this->amount = $amount;
        $this->month = (string) $month;
        $this->year = (int) $year;

        $this->dispatchBrowserEvent('openModal');
    }

    public function render()
    {
        $fees = Fee_assign::where('student_id', $this->studentId)->get();

        return view('livewire.user.fee.fee-collection', [
            'student' => $this->selectedStudent,
            'fees' => $fees,
            'totalBase' => $this->totalBase,
            'totalPaid' => $this->totalPaid,
            'pending' => $this->pending,
            'withFine' => $this->withFine,
            'percentPaid' => $this->percentPaid,
            'dueDate' => $this->dueDate,
        ]);
    }
}

