<?php

namespace App\Services;

use Chatify\ChatifyMessenger as BaseChatifyMessenger;
use App\Models\ChMessage as Message;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ChatifyMessenger extends BaseChatifyMessenger
{
    /**
     * Save a new message to database
     *
     * @param array $data
     * @return Message
     */
    public function newMessage($data)
    {
        $message = new Message();
        $message->from_id = $data['from_id'];
        $message->to_channel_id = $data['to_channel_id'];
        $message->body = $data['body'];
        $message->attachment = $data['attachment'];
        $message->reply_id = $data['reply_id'] ?? null;  // ✅ Include reply_id
        $message->save();
        return $message;
    }

    /**
     * Fetch & parse message with reply data
     *
     * @param Message $prefetchedMessage
     * @param int $id
     * @param bool $loadUserInfo
     * @return array
     */
    public function parseMessage($prefetchedMessage = null, $id = null, $loadUserInfo = true)
    {
        try {
            // Call parent method to get base message data
            $messageData = parent::parseMessage($prefetchedMessage, $id, $loadUserInfo);
        } catch (\Exception $e) {
            // If parent fails (e.g., no Auth), build basic message data
            $message = $prefetchedMessage ?? Message::find($id);
            if (!$message) {
                return [];
            }

            $messageData = [
                'id' => $message->id,
                'from_id' => $message->from_id,
                'to_channel_id' => $message->to_channel_id,
                'message' => $message->body,
                'attachment' => null,
                'created_at' => $message->created_at->toIso8601String(),
                'seen' => $message->seen,
            ];
        }

        // Add reply data if reply_id exists
        $message = $prefetchedMessage ?? Message::find($id);

        if ($message && $message->reply_id) {
            // Fetch original message with user data
            $replyMessage = Message::where('ch_messages.id', $message->reply_id)
                ->leftJoin('users', 'ch_messages.from_id', '=', 'users.id')
                ->leftJoin('admins', 'ch_messages.from_id', '=', 'admins.id')
                ->select(
                    'ch_messages.id',
                    'ch_messages.body',
                    'ch_messages.attachment',
                    DB::raw('COALESCE(users.name, admins.name) as user_name')
                )
                ->first();

            if ($replyMessage) {
                $messageData['reply_id'] = $replyMessage->id;
                $messageData['reply_body'] = $replyMessage->body;
                $messageData['reply_user_name'] = $replyMessage->user_name;

                // Parse attachment if exists
                $replyAttachment = null;
                if ($replyMessage->attachment) {
                    $attachmentOBJ = json_decode($replyMessage->attachment);
                    if ($attachmentOBJ) {
                        $replyAttachment = $attachmentOBJ->new_name;
                    }
                }
                $messageData['reply_attachment'] = $replyAttachment;
            } else {
                // Message was deleted or not found
                $messageData['reply_id'] = null;
                $messageData['reply_body'] = null;
                $messageData['reply_user_name'] = null;
                $messageData['reply_attachment'] = null;
            }
        } else {
            // No reply
            $messageData['reply_id'] = null;
            $messageData['reply_body'] = null;
            $messageData['reply_user_name'] = null;
            $messageData['reply_attachment'] = null;
        }

        return $messageData;
    }
}