<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ReferenceGenerator
{
    /**
     * EMISSÃO COMPLETA (recomendado):
     * - Aplica regra do dia 5 (mês seguinte) como data de expiração
     * - Se hoje for >= dia 6 → calcula multa e gera NOVA referência com (base + multa)
     * - Retorna payload completo para persistir/exibir
     *
     * @param string               $month        ex: 'January'
     * @param int                  $year         ex: 2025
     * @param string|object        $student      student_id (string) OU objeto Student
     * @param float                $baseAmount   valor base da taxa
     * @param array<string,mixed>  $options      ['fine_type'=>'fixed|percentage','fine_value'=>float,'entity'=>string,'expires_at'=>Date|str]
     * @return array{
     *    entity:string, reference:string,
     *    base_amount:float, fine_amount:float, total_amount:float,
     *    expires_at:\Carbon\Carbon, ttl_days:int,
     *    status:'pending'|'expired',
     *    month:string, year:int,
     *    should_expire_previous:bool,
     * }
     */
    public function issue(string $month, int $year, $student, float $baseAmount, array $options = []): array
    {
        // Entity
        $entity = $options['entity'] ?? config('payments.entity', '11111');

        // Student number (7 dígitos)
        if (is_object($student)) {
            $studentNumber = $this->createStudentCode($student);
        } else {
            $studentNumber = $this->sanitizeStudentId((string)$student);
        }

        // Mês (2 dígitos) para o algoritmo BCI
        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);

        // Cutoff: 5 do mês seguinte às 23:59:59
        $cutoff = $this->cutoffDay5OfNextMonth($month, $year);

        // Se foi passado expires_at explicitamente
        if (!empty($options['expires_at'])) {
            try {
                $cutoff = Carbon::parse($options['expires_at'])->setTime(23, 59, 59);
            } catch (\Throwable $e) {
                // mantém o cutoff padrão
            }
        }

        // Estamos depois do cutoff?
        $needsFine = now()->greaterThan($cutoff);
        $fineAmount = 0.0;

        // Cálculo de multa
        if ($needsFine) {
            $fineType = $options['fine_type'] ?? config('payments.fine.type', 'fixed');
            $fineValue = $options['fine_value'] ?? config('payments.fine.value', 0);
            $fineAmount = $this->calculateFine($baseAmount, $fineType, (float)$fineValue);
        }

        $totalAmount = round($baseAmount + $fineAmount, 2);

        // Se estamos a reemitir com multa, novo vencimento = próximo dia 5
        $expiresAt = $needsFine ? $this->nextCutoffFrom(now()) : $cutoff;

        // TTL em dias
        $ttlRaw = now()->diffInDays($expiresAt, false);
        $ttlDays = max(0, $ttlRaw);

        // Status
        $status = $ttlDays > 0 ? 'pending' : 'expired';

        // Geração da referência BCI com TOTAL (base+multa)
        $amountCents = (string)round($totalAmount * 100);
        $reference = $this->generateBCIReference($entity, $studentNumber, $monthNumber, $amountCents);

        // Flag para expirar referência anterior
        $shouldExpirePrevious = $needsFine;

        Log::info('Reference issued', [
            'entity' => $entity,
            'reference' => $reference,
            'month' => "$month $year",
            'amounts' => [
                'base' => $baseAmount,
                'fine' => $fineAmount,
                'total' => $totalAmount
            ],
            'expires_at' => $expiresAt->toDateTimeString(),
            'status' => $status
        ]);

        return [
            'entity' => $entity,
            'reference' => $reference,
            'base_amount' => $baseAmount,
            'fine_amount' => $fineAmount,
            'total_amount' => $totalAmount,
            'expires_at' => $expiresAt,
            'ttl_days' => $ttlDays,
            'status' => $status,
            'month' => $month,
            'year' => $year,
            'should_expire_previous' => $shouldExpirePrevious,
        ];
    }

    /**
     * Método simplificado para gerar apenas a referência
     */
    public function make(string $month, int $year, string $studentId, float $amount): string
    {
        $entity = config('payments.entity', '11111');
        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);
        $amountCents = (string)round($amount * 100);
        $studentNumber = $this->sanitizeStudentId($studentId);
        
        return $this->generateBCIReference($entity, $studentNumber, $monthNumber, $amountCents);
    }

    /**
     * Gerar referência a partir de objeto Student
     */
    public function makeFromStudent(string $month, int $year, $student, float $amount): string
    {
        $studentCode = $this->createStudentCode($student);
        return $this->make($month, $year, $studentCode, $amount);
    }

    /**
     * Algoritmo BCI/BMEPS para gerar referência
     * Implementação exata conforme especificação técnica do BCI
     * @return string A referência gerada (11 dígitos: número + mês + check digit)
     */
    private function generateBCIReference(string $entity, string $number, string $month, string $amountCents): string
    {
        // Limpar entrada - apenas dígitos
        $entity = preg_replace('/\D+/', '', $entity);
        $number = preg_replace('/\D+/', '', $number);
        $month = preg_replace('/\D+/', '', $month);
        $amountCents = preg_replace('/\D+/', '', $amountCents);

        // Formatação conforme documentação BCI
        // Nota: A documentação não especifica padding para entidade/valor na concatenação
        // mas o número deve ter 7 dígitos e o mês 2 dígitos para a referência final
        $number = str_pad($number, 7, '0', STR_PAD_LEFT);
        $month = str_pad($month, 2, '0', STR_PAD_LEFT);

        // Concatenar: entidade + número + mês + valor (sem padding adicional)
        $concat = $entity . $number . $month . $amountCents;

        // Algoritmo BCI para cálculo do check digit
        $Pi = 0;
        $n = strlen($concat);
        
        for ($i = 0; $i < $n; $i++) {
            $digit = (int)$concat[$i];
            $Si = $digit + $Pi;
            $Pi = ($Si * 10) % 97;
        }

        $Pn = ($Pi * 10) % 97;
        $checkDigit = 98 - $Pn;
        $checkDigitFormatted = str_pad((string)$checkDigit, 2, '0', STR_PAD_LEFT);

        // Referência final: número + mês + check digit
        $reference = $number . $month . $checkDigitFormatted;

        Log::debug('BCI Reference generated', [
            'entity' => $entity,
            'number' => $number,
            'month' => $month,
            'amount_cents' => $amountCents,
            'check_digit' => $checkDigitFormatted,
            'reference' => $reference
        ]);

        return $reference;
    }

    /**
     * Calcular multa
     */
    private function calculateFine(float $baseAmount, string $type, float $value): float
    {
        if ($value <= 0) {
            return 0.0;
        }

        if (strtolower($type) === 'percentage') {
            return round($baseAmount * ($value / 100), 2);
        }

        // tipo fixed
        return round($value, 2);
    }

    /**
     * Dia 5 do mês SEGUINTE ao da taxa
     */
    private function cutoffDay5OfNextMonth(string $month, int $year): Carbon
    {
        $monthNum = $this->getMonthNumber($month);
        return Carbon::create($year, $monthNum, 5)
            ->addMonthNoOverflow()
            ->setTime(23, 59, 59);
    }

    /**
     * Próximo dia 5 a partir de uma data
     */
    private function nextCutoffFrom(Carbon $from): Carbon
    {
        $candidate = Carbon::create($from->year, $from->month, 5, 23, 59, 59);
        
        if ($from->day() > 5) {
            $candidate = $candidate->addMonthNoOverflow();
        }
        
        return $candidate;
    }

    /**
     * Criar código único para o estudante (7 dígitos)
     */
    private function createStudentCode($student): string
    {
        // Extrair ID numérico
        $studentId = $student->student_id ?? $student->id ?? '';
        $numericPart = preg_replace('/[^0-9]/', '', (string)$studentId);
        
        // Extrair ano de nascimento
        $birthYear = $this->extractBirthYear($student);
        
        // Combinar: ID + últimos 2 dígitos do ano
        $yearPart = substr($birthYear, -2);
        $combined = $numericPart . $yearPart;

        // Ajustar para 7 dígitos
        if (strlen($combined) > 7) {
            $final = substr($combined, -7);
        } else {
            $final = str_pad($combined, 7, '0', STR_PAD_LEFT);
        }

        return $final;
    }

    /**
     * Extrair ano de nascimento do estudante
     */
    private function extractBirthYear($student): string
    {
        $dobFields = ['dob', 'date_of_birth', 'birth_date', 'birthday'];

        foreach ($dobFields as $field) {
            if (isset($student->$field) && !empty($student->$field)) {
                try {
                    $date = Carbon::parse($student->$field);
                    return $date->format('Y');
                } catch (\Exception $e) {
                    continue;
                }
            }
        }

        // Fallback para ano atual se não encontrar
        return date('Y');
    }

    /**
     * Sanitizar ID do estudante (garantir 7 dígitos)
     */
    private function sanitizeStudentId(string $studentId): string
    {
        // Extrair apenas números
        $numericOnly = preg_replace('/\D+/', '', $studentId);
        
        // Se não houver números, usar hash do ID original
        if (empty($numericOnly)) {
            $numericOnly = (string)abs(crc32($studentId));
        }
        
        // Garantir 7 dígitos
        if (strlen($numericOnly) > 7) {
            $sanitized = substr($numericOnly, -7);
        } else {
            $sanitized = str_pad($numericOnly, 7, '0', STR_PAD_LEFT);
        }

        return $sanitized;
    }

    /**
     * Converter nome do mês para número
     */
    private function getMonthNumber(string $monthName): int
    {
        $months = [
            // English
            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12,
            // Portuguese
            'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
            'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
            'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12,
            // Abbreviated
            'jan' => 1, 'feb' => 2, 'mar' => 3, 'apr' => 4,
            'may' => 5, 'jun' => 6, 'jul' => 7, 'aug' => 8,
            'sep' => 9, 'oct' => 10, 'nov' => 11, 'dec' => 12,
        ];

        $normalized = strtolower(trim($monthName));
        
        // Tentar encontrar o mês no array
        if (isset($months[$normalized])) {
            return $months[$normalized];
        }
        
        // Tentar interpretar como número
        if (is_numeric($monthName)) {
            $num = (int)$monthName;
            if ($num >= 1 && $num <= 12) {
                return $num;
            }
        }
        
        // Default para Janeiro se não reconhecer
        Log::warning('Month not recognized, defaulting to January', ['month' => $monthName]);
        return 1;
    }

    /**
     * Validar referência
     */
    public function validate(string $reference): bool
    {
        // Deve ter apenas dígitos
        if (!ctype_digit($reference)) {
            return false;
        }
        
        // Deve ter pelo menos 11 dígitos (7 do número + 2 do mês + 2 do check)
        if (strlen($reference) !== 11) {
            return false;
        }
        
        // Não pode ser tudo zeros
        if ($reference === str_repeat('0', strlen($reference))) {
            return false;
        }
        
        return true;
    }

    /**
     * Formatar referência para exibição
     */
    public function format(string $reference): string
    {
        $length = strlen($reference);

        if ($length === 11) {
            // Formato: XXX XXX XXX XX
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3) . ' ' .
                   substr($reference, 9, 2);
        }

        if ($length === 9) {
            // Formato: XXX XXX XXX
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3);
        }

        // Formato genérico para outros tamanhos
        $parts = [];
        for ($i = 0; $i < $length; $i += 3) {
            $parts[] = substr($reference, $i, 3);
        }
        
        return implode(' ', $parts);
    }

    /**
     * Teste com exemplo exato da documentação BCI
     * Deve retornar: 12122450702
     */
    public function testWithDocumentExample(): array
    {
        $entity = '54321';
        $number = '1212245';
        $month = '07';
        $amountCents = '200000'; // 2000,00 MT
        
        $reference = $this->generateBCIReference($entity, $number, $month, $amountCents);
        
        return [
            'input' => [
                'entity' => $entity,
                'number' => $number,
                'month' => $month,
                'amount_cents' => $amountCents,
            ],
            'reference' => $reference,
            'expected' => '12122450702',
            'test_passed' => $reference === '12122450702',
            'formatted' => $this->format($reference),
            'is_valid' => $this->validate($reference)
        ];
    }
}
