# Excel Marks Import System

This documentation covers the implementation of the Excel marks import functionality for the school ERP system.

## Overview

The Excel import system allows teachers and administrators to bulk import exam marks from Excel files (.xlsx/.xls) with automatic validation, duplicate handling, and comprehensive error reporting.

## Features

- **Multi-format Support**: Supports both .xlsx and .xls formats
- **Automatic Header Detection**: Recognizes various header formats (Portuguese/English)
- **Data Validation**: Validates marks range (0-20), student existence, and data integrity
- **Duplicate Handling**: Option to clean existing marks before import
- **Batch Processing**: Handles large files efficiently
- **Error Reporting**: Detailed error reporting with line-by-line feedback
- **Permission Control**: Role-based access control for import functionality
- **Preview Mode**: Shows preview before actual import
- **Progress Tracking**: Visual progress indicators during import

## Installation & Setup

### 1. Install Dependencies

```bash
composer require phpoffice/phpspreadsheet
```

### 2. Copy Service Files

Copy the following files to your Laravel application:

- `app/Services/ExcelMarksImportService.php`
- `app/Http/Livewire/Admin/Exams/ExcelImportMarks.php`
- `resources/views/livewire/admin/exams/excel-import-marks.blade.php`

### 3. Add Routes

Add the route to your `routes/web.php`:

```php
// Inside admin middleware group
Route::get('exams-grades/excel-import', \App\Http\Livewire\Admin\Exams\ExcelImportMarks::class)
    ->name('admin.excel.import.marks');
```

### 4. Run Migrations

```bash
php artisan migrate
```

### 5. Set Permissions

Ensure proper directory permissions:

```bash
chmod 755 storage/app/temp
mkdir -p storage/app/temp/excel-imports
chmod 755 storage/app/temp/excel-imports
```

## Excel File Format Requirements

### Supported Headers

The system recognizes the following header variations:

| Data Type | Supported Headers |
|-----------|------------------|
| Student Name | Nome, Name, Student_Name |
| Student ID | ID, Student_ID, ID_Estudante, ID do Estudante |
| ACS1 | ACS1, ASC1 |
| ACS2 | ACS2, ASC2 |
| MACS | MACS |
| AT | AT |
| MT | MT |
| NE/AF | NE, AF |

### Sample Excel Structure

```
| # | Nome              | ID   | ACS1 | ACS2 | AT   | NE   |
|---|-------------------|------|------|------|------|------|
| 1 | João Silva        | 1001 | 15.5 | 16.0 | 17.2 | 18.0 |
| 2 | Maria Santos      | 1002 | 14.0 | 15.5 | 16.0 | 17.5 |
| 3 | Pedro Oliveira    | 1003 | 18.0 | 17.5 | 19.0 | 19.5 |
```

### Validation Rules

- **Marks Range**: 0-20 (decimal values allowed)
- **Student Identification**: Either name or ID must be present
- **Number Format**: Accepts both comma (15,5) and dot (15.5) as decimal separators
- **Empty Cells**: Empty marks are ignored (not imported)
- **Invalid Values**: Values outside 0-20 range are ignored

## Usage Instructions

### Web Interface

1. **Navigate** to Admin → Exams → Excel Import
2. **Select Parameters**:
   - Academic Year
   - Trimester (1, 2, or 3)
   - Class Level
   - Classroom
   - Subject
3. **Upload File**: Choose your Excel file
4. **Preview**: Review the data before import
5. **Import**: Confirm and execute the import

### Command Line Interface

```bash
# Basic import
php artisan marks:import-excel /path/to/file.xlsx --year=2024 --trimester=1 --classroom="Turma A" --subject="Matemática"

# With options
php artisan marks:import-excel /path/to/file.xlsx --year=2024 --trimester=1 --classroom="Turma A" --subject="Matemática" --clean --dry-run

# Interactive mode (prompts for missing parameters)
php artisan marks:import-excel /path/to/file.xlsx
```

#### Command Options

- `--year`: Academic year (defaults to current year)
- `--trimester`: Trimester number (1, 2, or 3)
- `--classroom`: Classroom ID or name
- `--subject`: Subject ID or name
- `--clean`: Clean existing marks before import
- `--dry-run`: Preview changes without executing

## Permission System

### Role-Based Access

| Role | Permission | Description |
|------|------------|-------------|
| Admin (0) | Full Access | Can import for any class/subject |
| Accounting (1) | No Access | Cannot import marks |
| Teacher (2) | Conditional | Only if `canAddMarks = 'yes'` |
| Academic Registrar (3) | Full Access | Can import for any class/subject |

### Permission Configuration

Permissions are controlled via the `admin.permission` route:

```php
// Enable teacher mark addition
Admin::where('role', 2)->update(['canAddMarks' => 'yes']);

// Enable academic registrar mark editing
Admin::where('role', 3)->update(['canEditMarks' => 'yes']);
```

## Error Handling

### Common Errors and Solutions

| Error | Cause | Solution |
|-------|-------|----------|
| "Student not found" | Student name/ID doesn't match database | Verify student exists in the classroom |
| "Excel validation failed" | Invalid file format | Use .xlsx or .xls format |
| "Field year is required" | Missing required parameter | Provide all required fields |
| "Invalid trimester" | Trimester not 1, 2, or 3 | Use valid trimester number |
| "Subject not found" | Subject ID doesn't exist | Verify subject exists |
| "Classroom not found" | Classroom ID doesn't exist | Verify classroom exists |

### Error Logging

All import operations are logged with detailed information:

```php
// Check import logs
tail -f storage/logs/laravel.log | grep "Excel import"
```

## Testing

### Running Tests

```bash
# Run all Excel import tests
php artisan test --filter=ExcelMarksImport

# Run specific test class
php artisan test tests/Feature/ExcelMarksImportTest.php

# Run unit tests only
php artisan test tests/Unit/ExcelMarksImportServiceTest.php
```

### Test Coverage

- File format validation
- Mark range validation
- Student matching logic
- Permission checking
- Error handling
- Data integrity
- Duplicate detection
- Calculation accuracy (MACS, MT)

## Performance Considerations

### Memory Management

For large files (>1000 rows):

```php
// Increase memory limit in .env
EXCEL_IMPORT_MEMORY_LIMIT=1024M

// Set timeout for long imports
EXCEL_IMPORT_TIMEOUT=600
```

### Batch Processing

The system processes imports in batches to prevent timeouts:

- Default batch size: 100 records
- Automatic memory cleanup between batches
- Progress tracking for large imports

## Troubleshooting

### Common Issues

1. **File Upload Fails**
   - Check file size (max 10MB)
   - Verify file permissions
   - Check available disk space

2. **Import Timeout**
   - Increase `EXCEL_IMPORT_TIMEOUT`
   - Process smaller batches
   - Check server resources

3. **Memory Errors**
   - Increase `EXCEL_IMPORT_MEMORY_LIMIT`
   - Use command line for large files
   - Clear cache before import

4. **Permission Denied**
   - Check user role permissions
   - Verify `canAddMarks` setting
   - Check route middleware

### Debug Mode

Enable debug logging:

```php
// Add to .env
LOG_LEVEL=debug
EXCEL_IMPORT_DEBUG=true
```

## Maintenance

### Regular Tasks

1. **Clean Temporary Files**
   ```bash
   php artisan schedule:run
   # Or manually: rm -rf storage/app/temp/excel-imports/*
   ```

2. **Monitor Import Logs**
   ```bash
   tail -f storage/logs/laravel.log | grep "Excel import"
   ```

3. **Database Optimization**
   ```sql
   OPTIMIZE TABLE exams;
   ```

## Security Considerations

- File type validation prevents malicious uploads
- Path traversal protection
- SQL injection prevention through Eloquent ORM
- Role-based access control
- Input sanitization and validation
- Temporary file cleanup after processing

## API Integration

The service can be integrated with external systems:

```php
use App\Services\ExcelMarksImportService;

$service = new ExcelMarksImportService();
$result = $service->importMarks($filePath, $importData);

if ($result['success']) {
    // Handle success
    $stats = $result['data'];
} else {
    // Handle error
    $error = $result['message'];
}
```

## Support

For issues or questions:

1. Check the error logs
2. Verify file format compliance
3. Test with sample data
4. Review permission settings
5. Contact system administrator

---

*Last updated: May 28, 2025*