<div class="page-content">
    <div class="main-wrapper">
        <div>
            <!-- Page Header with Filter -->
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px;">
                <div>
                    <h2 style="margin: 0; font-size: 24px; color: #333; font-weight: 600;">Pagamentos e Taxas</h2>
                    <p style="margin: 5px 0 0 0; color: #666; font-size: 14px;">Acompanhe suas taxas escolares e pagamentos</p>
                </div>
                <div>
                    <label for="yearFilterUser" style="display: block; font-size: 12px; margin-bottom: 5px; color: #666; font-weight: 500;">Ano Letivo</label>
                    <select id="yearFilterUser" class="form-select" wire:model="selectedYear" style="min-width: 140px; padding: 8px 12px; border-radius: 8px; border: 1px solid #ddd;">
                        @for ($i = now()->year; $i >= now()->year - 5; $i--)
                            <option value="{{ $i }}">{{ $i }}</option>
                        @endfor
                    </select>
                </div>
            </div>

            @php
                // Verifica se os dados do service unificado estão disponíveis
                $hasMonthlyData = isset($monthlyData) && is_array($monthlyData) && count($monthlyData) > 0;
                
                // Se tem dados do service, usa eles
                if ($hasMonthlyData) {
                    $totals = [
                        'expected' => $totalExpected ?? 0,
                        'paid' => $totalPaid ?? 0,
                        'pending' => $totalPending ?? 0,
                        'overdue' => $totalOverdue ?? 0,
                        'fines' => $totalFines ?? 0,
                        'general' => ($totalExpected ?? 0) + ($totalFines ?? 0)
                    ];
                } else {
                    // Fallback para dados antigos (se existir)
                    $currentYear = now()->year;
                    $months = ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
                              'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];

                    $totals = [
                        'expected' => 0,
                        'paid' => 0,
                        'pending' => 0,
                        'overdue' => 0,
                        'fines' => 0,
                        'general' => 0
                    ];

                    // Gera dados básicos como fallback
                    $monthlyData = [];
                    $baseFeePerMonth = 4148.00; // Valor padrão aproximado
                    
                    foreach ($months as $index => $month) {
                        $isCurrentOrPast = ($index + 1) <= now()->month;
                        $dueDate = now()->year . '-' . str_pad($index + 1, 2, '0', STR_PAD_LEFT) . '-05';
                        $isPastDue = $isCurrentOrPast && now()->gt($dueDate);
                        
                        $monthlyData[] = [
                            'month' => $month,
                            'year' => $currentYear,
                            'base_amount' => $baseFeePerMonth,
                            'fine_amount' => $isPastDue ? ($baseFeePerMonth * 0.05) : 0, // 5% de multa
                            'total_expected' => $baseFeePerMonth + ($isPastDue ? ($baseFeePerMonth * 0.05) : 0),
                            'is_paid' => false,
                            'is_overdue' => $isPastDue,
                            'due_date' => \Carbon\Carbon::parse($dueDate),
                            'existing_payment' => null,
                            'fee_names' => 'Taxa Escolar',
                            'data_status' => $isPastDue ? 'overdue' : 'pending'
                        ];
                        
                        $totals['expected'] += $baseFeePerMonth;
                        if ($isPastDue) {
                            $totals['overdue'] += $baseFeePerMonth;
                            $totals['fines'] += ($baseFeePerMonth * 0.05);
                        } else {
                            $totals['pending'] += $baseFeePerMonth;
                        }
                    }
                    
                    $totals['general'] = $totals['expected'] + $totals['fines'];
                }

                $paidPercentage = $totals['expected'] > 0 ? round(($totals['paid'] / $totals['expected']) * 100, 1) : 0;
            @endphp

            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card">
                    <div class="summary-card-icon card-total">
                        <i class="fas fa-file-invoice-dollar"></i>
                    </div>
                    <h3>Total de Taxas</h3>
                    <div class="amount">{{ number_format($totals['expected'], 2, ',', '.') }} MT</div>
                    <p class="description">Ano letivo {{ $currentYear ?? date('Y') }}</p>
                </div>
                
                <div class="summary-card">
                    <div class="summary-card-icon card-paid">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h3>Já Pago</h3>
                    <div class="amount">{{ number_format($totals['paid'], 2, ',', '.') }} MT</div>
                    <p class="description">{{ $paidPercentage }}% do total</p>
                </div>
                
                <div class="summary-card">
                    <div class="summary-card-icon card-pending">
                        <i class="fas fa-clock"></i>
                    </div>
                    <h3>Pendente</h3>
                    <div class="amount">{{ number_format($totals['pending'], 2, ',', '.') }} MT</div>
                    <p class="description">A pagar</p>
                </div>
                
                <div class="summary-card">
                    <div class="summary-card-icon card-overdue">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <h3>Em Atraso</h3>
                    <div class="amount">{{ number_format($totals['overdue'], 2, ',', '.') }} MT</div>
                    <p class="description">Com multas</p>
                </div>
            </div>

            <!-- Fee Table Section -->
            <div class="fee-section">
                <div class="section-header">
                    <h2>
                        <i class="fas fa-list"></i>
                        Detalhes das Taxas - {{ $currentYear ?? date('Y') }}
                    </h2>
                    <div class="fee-filters">
                        <button class="filter-btn active" data-filter="all">Todas</button>
                        <button class="filter-btn" data-filter="pending">Pendentes</button>
                        <button class="filter-btn" data-filter="paid">Pagas</button>
                        <button class="filter-btn" data-filter="overdue">Em Atraso</button>
                    </div>
                </div>

                <div class="fee-table-container">
                    @if(count($monthlyData) > 0)
                        <table class="table table-striped table-hover fee-table">
                            <thead>
                                <tr>
                                    <th>Grupo de Taxas</th>
                                    <th>Código</th>
                                    <th>Vencimento</th>
                                    <th>Estado</th>
                                    <th>Montante Base</th>
                                    <th>Desconto</th>
                                    <th>Multa <i class="fas fa-info-circle" title="Multa preservada no histórico"></i></th>
                                    <th>Total a Pagar</th>
                                    <th>Pago</th>
                                    <th>Saldo</th>
                                    <th>Data</th>
                                    <th>Ação</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($monthlyData as $monthData)
                                @php
                                    // Normaliza dados para compatibilidade
                                    $month = $monthData['month'] ?? 'N/A';
                                    $monthPt = $month; // Mês já está em português
                                    $year = $monthData['year'] ?? date('Y');
                                    $baseAmount = $monthData['base_amount'] ?? 0;
                                    $fineAmount = $monthData['fine_amount'] ?? 0;
                                    $discountApplied = $monthData['discount_applied'] ?? 0;
                                    $totalExpected = $monthData['total_expected'] ?? ($baseAmount + $fineAmount - $discountApplied);
                                    $isPaid = $monthData['is_paid'] ?? false;
                                    $isOverdue = $monthData['is_overdue'] ?? false;
                                    $paidAmount = $isPaid ? ($monthData['paid_amount'] ?? $totalExpected) : 0;
                                    $balanceAmount = $isPaid ? 0 : $totalExpected;
                                    $paymentDate = $isPaid ? ($monthData['payment_date'] ?? 'N/A') : 'N/A';
                                    $dueDate = isset($monthData['due_date']) ? $monthData['due_date']->format('d/m/Y') : 'N/A';
                                    $feeNames = $monthData['fee_names'] ?? 'Taxa Escolar';
                                    $dataStatus = $monthData['data_status'] ?? ($isPaid ? 'paid' : ($isOverdue ? 'overdue' : 'pending'));
                                    
                                    // Status display
                                    if ($isPaid) {
                                        $statusBadge = '<span class="badge bg-success"><i class="fas fa-check"></i> Pago</span>';
                                    } elseif ($isOverdue) {
                                        $statusBadge = '<span class="badge bg-danger"><i class="fas fa-exclamation-triangle"></i> Em Atraso</span>';
                                    } else {
                                        $statusBadge = '<span class="badge bg-warning"><i class="fas fa-clock"></i> Não Pago</span>';
                                    }

                                    // Busca referência existente para este mês
                                    $hasReference = false;
                                    $referenceData = null;
                                    if (isset($payment_references) && is_iterable($payment_references)) {
                                        foreach ($payment_references as $ref) {
                                            if ($ref->fee_month === $month && $ref->fee_year == $year) {
                                                $hasReference = true;
                                                $referenceData = $ref;
                                                break;
                                            }
                                        }
                                    }
                                @endphp

                                <tr class="{{ $isOverdue && !$isPaid ? 'row-overdue' : '' }}" data-status="{{ $dataStatus }}">
                                    <td data-label="Mês/Ano">
                                        {{ $monthPt }} {{ $year }}
                                        @if($isOverdue)
                                            <br><small class="text-warning">
                                                <i class="fas fa-clock"></i> Em atraso
                                            </small>
                                        @endif
                                    </td>
                                    <td data-label="Código">{{ $feeNames }}</td>
                                    <td data-label="Vencimento">{{ $dueDate }}</td>
                                    <td data-label="Estado">{!! $statusBadge !!}</td>
                                    <td data-label="Montante Base">
                                        <span class="fw-bold">{{ number_format($baseAmount, 2, ',', '.') }} MT</span>
                                    </td>
                                    <td data-label="Desconto">
                                        @if($discountApplied > 0)
                                            <span class="text-success fw-bold">-{{ number_format($discountApplied, 2, ',', '.') }} MT</span>
                                        @else
                                            <span class="text-muted">0,00 MT</span>
                                        @endif
                                    </td>
                                    <td data-label="Multa">
                                        @if($fineAmount > 0)
                                            <span class="text-danger fw-bold">
                                                {{ number_format($fineAmount, 2, ',', '.') }} MT
                                            </span>
                                        @else
                                            <span class="text-muted">0,00 MT</span>
                                        @endif
                                    </td>
                                    <td data-label="Total a Pagar">
                                        <strong>{{ number_format($totalExpected, 2, ',', '.') }} MT</strong>
                                    </td>
                                    <td data-label="Pago">
                                        @if($isPaid)
                                            <span class="text-success fw-bold">{{ number_format($paidAmount, 2, ',', '.') }} MT</span>
                                        @else
                                            <span class="text-muted">0,00 MT</span>
                                        @endif
                                    </td>
                                    <td data-label="Saldo">
                                        @if($balanceAmount > 0)
                                            <span class="text-danger fw-bold">{{ number_format($balanceAmount, 2, ',', '.') }} MT</span>
                                        @else
                                            <span class="text-success">0,00 MT</span>
                                        @endif
                                    </td>
                                    <td data-label="Data">{{ $paymentDate }}</td>
                                    <td data-label="Ação">
                                        <div class="action-buttons-container">
                                            @if(!$isPaid)
                                                <button
                                                    onclick="openPaymentModal('{{ $month }}', {{ $year }}, {{ $baseAmount }}, {{ $fineAmount }}, {{ $totalExpected }}, '{{ $feeNames }}', {{ $discountApplied }})"
                                                    class="btn btn-primary btn-sm">
                                                    <i class="fas fa-credit-card me-1"></i>Pagar
                                                </button>
                                                
                                                @if($hasReference && $referenceData)
                                                    <button
                                                        onclick="showReferenceModal('{{ $month }}', {{ $year }}, '{{ $referenceData->entity_code ?? 'N/A' }}', '{{ $referenceData->reference_number ?? 'N/A' }}', '{{ number_format($referenceData->amount ?? $totalExpected, 2, ',', '.') }} MT', '{{ $referenceData->expires_at ? \Carbon\Carbon::parse($referenceData->expires_at)->format('d/m/Y H:i') : 'N/A' }}')"
                                                        class="btn btn-info btn-sm ms-1"
                                                        title="Ver dados da referência">
                                                        <i class="fas fa-barcode me-1"></i>Ver Ref.
                                                    </button>
                                                @endif
                                            @else
                                                <div class="d-flex align-items-center justify-content-center gap-2">
                                                    <span class="status-badge status-paid">
                                                        <i class="fas fa-check-circle me-1"></i>Pago
                                                    </span>

                                                    @php
                                                        // Buscar dados do pagamento (Fee_assign para Cash/POS ou PaymentReference para Referência)
                                                        $feeAssignPayment = $monthData['existing_payment'] ?? $monthData['payment_data'] ?? null;
                                                    @endphp

                                                    @if($hasReference && $referenceData)
                                                        {{-- Pagamento via Referência --}}
                                                        <button
                                                            wire:click="downloadReceipt('{{ $referenceData->id }}')"
                                                            class="btn btn-success btn-sm"
                                                            title="Baixar comprovativo de pagamento">
                                                            <i class="fas fa-file-download me-1"></i>Comprovativo
                                                        </button>
                                                        <button
                                                            wire:click="downloadOfficialReceipt('{{ $referenceData->id }}')"
                                                            class="btn btn-primary btn-sm"
                                                            title="Baixar recibo oficial">
                                                            <i class="fas fa-receipt me-1"></i>Recibo
                                                        </button>
                                                        <button
                                                            onclick="showReferenceModal('{{ $month }}', {{ $year }}, '{{ $referenceData->entity_code ?? 'N/A' }}', '{{ $referenceData->reference_number ?? 'N/A' }}', '{{ number_format($referenceData->amount ?? $totalExpected, 2, ',', '.') }} MT', '{{ $referenceData->expires_at ? \Carbon\Carbon::parse($referenceData->expires_at)->format('d/m/Y H:i') : 'N/A' }}')"
                                                            class="btn btn-info btn-sm"
                                                            title="Ver dados da referência">
                                                            <i class="fas fa-barcode me-1"></i>Ver Ref.
                                                        </button>
                                                    @elseif($feeAssignPayment)
                                                        {{-- Pagamento via Cash/POS --}}
                                                        <button
                                                            wire:click="downloadCashPosReceipt('{{ $feeAssignPayment->id ?? '' }}')"
                                                            class="btn btn-success btn-sm"
                                                            title="Baixar recibo de pagamento">
                                                            <i class="fas fa-receipt me-1"></i>Recibo
                                                        </button>
                                                    @endif
                                                </div>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                            <tfoot class="table-light">
                                <tr>
                                    <th colspan="4">Totais:</th>
                                    <th>{{ number_format(collect($monthlyData)->sum('base_amount'), 2, ',', '.') }} MT</th>
                                    <th>{{ number_format(collect($monthlyData)->sum(function($m) { return $m['discount_applied'] ?? 0; }), 2, ',', '.') }} MT</th>
                                    <th>{{ number_format(collect($monthlyData)->sum('fine_amount'), 2, ',', '.') }} MT</th>
                                    <th>{{ number_format(collect($monthlyData)->sum('total_expected'), 2, ',', '.') }} MT</th>
                                    <th>{{ number_format(collect($monthlyData)->where('is_paid', true)->sum(function($m) { return $m['paid_amount'] ?? $m['total_expected']; }), 2, ',', '.') }} MT</th>
                                    <th>{{ number_format(collect($monthlyData)->where('is_paid', false)->sum('total_expected'), 2, ',', '.') }} MT</th>
                                    <th colspan="2">
                                        <small class="text-muted">
                                            {{ collect($monthlyData)->where('is_paid', true)->count() }} de {{ count($monthlyData) }} meses pagos
                                        </small>
                                    </th>
                                </tr>
                            </tfoot>
                        </table>

                        {{-- Informação sobre Service
                        @if(!$hasMonthlyData)
                        <div class="alert alert-warning mt-3">
                            <h6><i class="fas fa-exclamation-triangle"></i> Dados de Fallback</h6>
                            <p class="mb-0">Os dados mostrados são calculados automaticamente. Para obter cálculos precisos de multas baseados na sua estrutura de taxa.</p>
                        </div>
                        @else
                        <div class="alert alert-success mt-3">
                            <h6><i class="fas fa-check-circle"></i> Service Unificado Ativo</h6>
                            <p class="mb-0">As multas são calculadas automaticamente baseado na sua estrutura de taxa e preservadas no histórico para auditoria completa.</p>
                        </div>
                        @endif --}}

                    @else
                        <div class="text-center py-4">
                            <i class="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
                            <h5>Não foi possível carregar os dados das taxas</h5>
                            <p class="text-muted">Verifique se Estrutura de Taxas está configurado corretamente.</p>
                            <button class="btn btn-primary" onclick="window.location.reload()">
                                <i class="fas fa-sync"></i> Recarregar Página
                            </button>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="quick-actions">
                <a href="{{ route('user.statement.download') }}" class="action-btn" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none;">
                    <i class="fas fa-file-pdf" style="color: white;"></i>
                    <h4 style="color: white;">Baixar Histórico de Pagamentos</h4>
                    <p style="color: rgba(255,255,255,0.9);">Relatório completo em PDF com todos os pagamentos</p>
                </a>
            </div>
        </div>
    </div>

    <!-- Payment Modal (mantido igual) -->
    <div class="modal fade" id="collectFeeModal" tabindex="-1" role="dialog" aria-labelledby="collectFeeModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <form id="paymentForm" onsubmit="submitPayment(event)">
                    <div class="modal-header bg-primary text-white">
                        <h5 class="modal-title" id="collectFeeModalLabel">
                            <i class="fas fa-credit-card me-2"></i>Efetuar Pagamento
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Fechar"></button>
                    </div>
                    
                    <div class="modal-body">
                        <!-- Fee Details Display -->
                        <div class="fee-details-card mb-4">
                            <div class="card bg-light">
                                <div class="card-body">
                                    <h6 class="card-title text-primary">
                                        <i class="fas fa-info-circle me-2"></i>Detalhes da Taxa
                                    </h6>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <p><strong>Período:</strong> <span id="modalPeriod">-</span></p>
                                            <p><strong>Taxa:</strong> <span id="modalFeeName">-</span></p>
                                        </div>
                                        <div class="col-md-6">
                                            <p><strong>Valor Base:</strong> <span id="modalBaseAmount">0,00 MT</span></p>
                                            <p><strong>Multa Calculada:</strong> <span id="modalCalculatedFine" class="text-danger">0,00 MT</span></p>
                                        </div>
                                    </div>
                                    <hr>
                                    <p class="mb-0"><strong>Total a Pagar:</strong> <span id="modalTotalAmount" class="text-success fs-5">0,00 MT</span></p>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Form -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="amount" class="form-label">
                                        <i class="fas fa-money-bill-wave me-1"></i>Montante a Pagar *
                                    </label>
                                    <div class="input-group">
                                        <input type="number"
                                               class="form-control"
                                               id="amount"
                                               name="amount"
                                               step="0.01"
                                               min="0"
                                               readonly
                                               required>
                                        <span class="input-group-text">MT</span>
                                    </div>
                                    <small class="text-muted">Valor fixo (base + multa aplicável)</small>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="fine" class="form-label">
                                        <i class="fas fa-exclamation-triangle me-1"></i>Multa por Atraso
                                    </label>
                                    <div class="input-group">
                                        <input type="number"
                                               class="form-control"
                                               id="fine"
                                               name="fine"
                                               step="0.01"
                                               min="0"
                                               readonly>
                                        <span class="input-group-text">MT</span>
                                    </div>
                                    <small class="text-muted">Multa calculada automaticamente</small>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="paymentMode" class="form-label">
                                        <i class="fas fa-credit-card me-1"></i>Modo de Pagamento *
                                    </label>
                                    <select class="form-control" id="paymentMode" name="paymentMode" required>
                                        <option value="Reference">Entidade e Referência</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="note" class="form-label">
                                        <i class="fas fa-sticky-note me-1"></i>Observações
                                    </label>
                                    <textarea class="form-control"
                                              id="note"
                                              name="note"
                                              rows="3"
                                              placeholder="Observações (opcional)"></textarea>
                                </div>
                            </div>
                        </div>

                        <!-- Hidden fields -->
                        <input type="hidden" id="hiddenMonth" name="month">
                        <input type="hidden" id="hiddenYear" name="year">
                        <input type="hidden" id="hiddenDiscount" name="discount" value="0">

                        <!-- Payment Summary -->
                        <div class="payment-summary mt-3">
                            <div class="card border-success">
                                <div class="card-body bg-light">
                                    <h6 class="text-success mb-3">
                                        <i class="fas fa-calculator me-2"></i>Resumo do Pagamento
                                    </h6>
                                    <div class="row text-center">
                                        <div class="col-4">
                                            <div class="border-end">
                                                <div class="text-muted small">Valor Base</div>
                                                <div class="fw-bold" id="summaryBase">0,00 MT</div>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="border-end">
                                                <div class="text-muted small">Multa</div>
                                                <div class="fw-bold text-warning" id="summaryFine">0,00 MT</div>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="text-muted small">Total a Pagar</div>
                                            <div class="fw-bold text-success fs-5" id="summaryTotal">0,00 MT</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times me-1"></i>Cancelar
                        </button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-check me-1"></i>Confirmar Pagamento
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Reference Modal (mantido igual) -->
    <div class="modal fade" id="referenceModal" tabindex="-1" role="dialog" aria-labelledby="referenceModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-md" role="document">
            <div class="modal-content">
                <div class="modal-header bg-info text-white">
                    <h5 class="modal-title" id="referenceModalLabel">
                        <i class="fas fa-barcode me-2"></i>Dados da Referência
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="reference-details">
                        <div class="card bg-light mb-3">
                            <div class="card-body">
                                <h6 class="card-title text-info mb-3">
                                    <i class="fas fa-info-circle me-2"></i>Detalhes do Pagamento
                                </h6>
                                <div class="row">
                                    <div class="col-12 mb-2">
                                        <strong>Período:</strong> <span id="refPeriod" class="text-dark">-</span>
                                    </div>
                                    <div class="col-12 mb-2">
                                        <strong>Valor:</strong> <span id="refAmount" class="text-success fw-bold">-</span>
                                    </div>
                                    <div class="col-12 mb-2">
                                        <strong>Validade:</strong> <span id="refExpires" class="text-warning">-</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="reference-payment-data">
                            <div class="alert alert-info">
                                <h6 class="alert-heading">
                                    <i class="fas fa-university me-2"></i>Dados para Pagamento
                                </h6>
                                <hr>
                                <div class="payment-info">
                                    <div class="row">
                                        <div class="col-6">
                                            <div class="payment-field">
                                                <label class="form-label fw-bold">Entidade:</label>
                                                <div class="payment-value" id="refEntity">-</div>
                                            </div>
                                        </div>
                                        <div class="col-6">
                                            <div class="payment-field">
                                                <label class="form-label fw-bold">Referência:</label>
                                                <div class="payment-value" id="refReference">-</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="reference-instructions mt-3">
                            <div class="card border-success">
                                <div class="card-body">
                                    <h6 class="card-title text-success">
                                        <i class="fas fa-lightbulb me-2"></i>Como pagar
                                    </h6>
                                    <ul class="list-unstyled mb-0">
                                        <li class="mb-2">
                                            <i class="fas fa-mobile-alt text-primary me-2"></i>
                                            <strong>M-Pesa:</strong> Use a entidade e referência no menu de pagamentos
                                        </li>
                                        <li class="mb-2">
                                            <i class="fas fa-university text-success me-2"></i>
                                            <strong>Balcão:</strong> Apresente estes dados em qualquer banco ou casa de câmbio
                                        </li>
                                        <li class="mb-0">
                                            <i class="fas fa-globe text-info me-2"></i>
                                            <strong>Internet Banking:</strong> Use a entidade e referência na área de pagamentos
                                        </li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Fechar
                    </button>
                    <button type="button" class="btn btn-primary" onclick="copyReferenceData()">
                        <i class="fas fa-copy me-1"></i>Copiar Dados
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Estilos mantidos iguais --}}
<style>
/* Optimized CSS - Combined and minified */
.summary-cards{display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:20px;margin-bottom:30px}
.summary-card{background:#fff;padding:25px;border-radius:15px;box-shadow:0 5px 20px rgba(0,0,0,.08);transition:transform .3s ease}
.summary-card:hover{transform:translateY(-5px)}
.summary-card-icon{width:50px;height:50px;border-radius:12px;display:flex;align-items:center;justify-content:center;font-size:24px;margin-bottom:15px}
.card-total{background:#e3f2fd;color:#1976d2}
.card-paid{background:#e8f5e9;color:#388e3c}
.card-pending{background:#fff3e0;color:#f57c00}
.card-overdue{background:#ffebee;color:#d32f2f}
.summary-card h3{font-size:14px;color:#666;font-weight:500;margin-bottom:5px}
.summary-card .amount{font-size:28px;font-weight:700;color:#333;margin-bottom:5px}
.summary-card .description{font-size:12px;color:#999}
.fee-section{background:#fff;border-radius:15px;padding:25px;box-shadow:0 5px 20px rgba(0,0,0,.08);margin-bottom:30px}
.section-header{display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;padding-bottom:15px;border-bottom:2px solid #f0f2f5}
.section-header h2{font-size:20px;color:#333;margin:0;display:flex;align-items:center;gap:10px}
.fee-filters{display:flex;gap:10px}
.filter-btn{padding:8px 16px;border:1px solid #ddd;border-radius:8px;background:#fff;cursor:pointer;transition:all .3s ease;font-size:14px;-webkit-tap-highlight-color:transparent;touch-action:manipulation;user-select:none;-webkit-user-select:none}
.filter-btn:active{transform:scale(.95);background:#f0f0f0}
.filter-btn.active{background:#667eea;color:#fff;border-color:#667eea}
.fee-table-container{overflow-x:auto}
.fee-table{width:100%;border-collapse:collapse}
.fee-table th{text-align:left;padding:12px;font-size:13px;font-weight:600;color:#666;border-bottom:2px solid #f0f2f5}
.fee-table td{padding:16px 12px;border-bottom:1px solid #f0f2f5}
.fee-table tr:hover{background-color:#f8f9fa}
.quick-actions{display:grid;grid-template-columns:1fr;gap:15px;margin-top:30px}
.action-btn{background:#fff;border:2px solid #e0e0e0;padding:25px;border-radius:12px;text-align:center;cursor:pointer;transition:all .3s ease;text-decoration:none;display:block;box-shadow:0 2px 8px rgba(0,0,0,0.08)}
.action-btn:hover{transform:translateY(-3px);box-shadow:0 6px 20px rgba(102,126,234,0.3)}
.action-btn i{font-size:32px;color:#667eea;margin-bottom:10px;display:block}
.action-btn h4{font-size:16px;margin-bottom:5px;color:#333}
.action-btn p{font-size:13px;color:#666;margin:0}
.text-muted{color:#999!important;font-size:13px}
.form-control[readonly]{background-color:#f8f9fa;border-color:#dee2e6;color:#6c757d;cursor:not-allowed}
.form-control[readonly]:focus{background-color:#f8f9fa;border-color:#dee2e6;box-shadow:none}
.fee-details-card .card{border:1px solid #e3f2fd}
.payment-summary .card{border:1px solid #c8e6c9}
.modal-header.bg-primary{background:linear-gradient(135deg,#667eea 0%,#764ba2 100%)!important}
.modal-header.bg-info{background:linear-gradient(135deg,#17a2b8 0%,#138496 100%)!important}
.form-label{font-weight:600;color:#495057}
.input-group-text{background-color:#f8f9fa;border-color:#ced4da}
.modal-loading-overlay{position:absolute;top:0;left:0;right:0;bottom:0;background:rgba(255,255,255,.9);display:none;align-items:center;justify-content:center;z-index:9999;flex-direction:column;gap:15px}
.modal-loading-overlay.active{display:flex}
.modal-loading-spinner{width:50px;height:50px;border:4px solid #f3f3f3;border-top:4px solid #667eea;border-radius:50%;animation:spin 1s linear infinite}
@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}

/* Action buttons container */
.action-buttons-container{display:flex;flex-wrap:wrap;gap:5px;align-items:center}
.action-buttons-container .btn{white-space:nowrap;font-size:13px}

/* Status badge - Same as admin */
.status-badge{padding:6px 12px;border-radius:20px;font-size:11px;font-weight:600;display:inline-flex;align-items:center;gap:5px}
.status-paid{background:#e8f5e9;color:#2e7d32}
.status-pending{background:#e3f2fd;color:#1565c0}
.status-overdue{background:#fff3e0;color:#f57c00}
.gap-2{gap:0.5rem}

/* Reference modal styles */
.reference-details .payment-field{margin-bottom:15px}
.reference-details .payment-value{background:#f8f9fa;padding:10px;border-radius:6px;font-family:monospace;font-size:16px;font-weight:bold;color:#495057;border:1px solid #e9ecef}
.reference-payment-data .alert{margin-bottom:0}
.reference-instructions .card{background:#f8fff8}

/* Remover cores de fundo das linhas - apenas badges mostram status */
.fee-table tbody tr {
    background-color: transparent !important;
}

/* Zebra striping neutro - cinza claro alternado */
.fee-table.table-striped tbody tr:nth-of-type(odd) {
    background-color: #f8f9fa !important;
}

.fee-table.table-striped tbody tr:nth-of-type(even) {
    background-color: #ffffff !important;
}

/* Hover effect uniforme e sutil para todas as linhas */
.fee-table tbody tr:hover {
    background-color: #e9ecef !important;
}

.fw-bold {
    font-weight: bold;
}

.bg-success { background-color: #28a745 !important; }
.bg-warning { background-color: #ffc107 !important; }
.bg-danger { background-color: #dc3545 !important; }

/* Responsive Design */
@media (max-width:992px){
    .page-content{padding:15px 10px}
    .summary-cards{grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:15px}
    .summary-card{padding:20px}
    .fee-section{padding:20px 15px}
    .section-header{flex-direction:column;align-items:flex-start;gap:15px}
    .fee-filters{width:100%;overflow-x:auto;padding-bottom:5px}
    .filter-btn{flex-shrink:0;padding:6px 12px;font-size:13px;min-width:60px;height:35px;display:inline-flex;align-items:center;justify-content:center}
    .fee-table thead{display:none!important}
    .fee-table tbody tr{display:block!important;margin-bottom:15px;border:1px solid #e9ecef;border-radius:10px;padding:15px;background:#fff;box-shadow:0 2px 10px rgba(0,0,0,.05)}
        .fee-table tbody td{display:flex!important;justify-content:space-between;align-items:center;padding:8px 0;border:none!important;border-bottom:1px solid #f1f3f4!important;white-space:normal;word-break:break-word}
            .fee-table tbody td:last-child{border-bottom:none!important}
            .fee-table tbody td::before{content:attr(data-label);font-weight:600;color:#495057;flex:0 0 35%;margin-right:10px;font-size:.8rem}
                .fee-table tbody td .badge,.fee-table tbody td .btn{margin-left:auto}
                .fee-table tbody td .action-buttons-container{margin-left:auto;flex-direction:column;gap:3px}
                .fee-table tbody tr.filter-hidden{display:none!important;height:0!important;margin:0!important;padding:0!important;border:none!important}
                .quick-actions{grid-template-columns:1fr;gap:10px}
                .action-btn{padding:20px 10px}
                .modal-dialog{margin:10px;max-width:calc(100% - 20px)}
            }
            @media (max-width:576px){
                .summary-cards{grid-template-columns:1fr}
                .fee-table tbody td{flex-direction:column;align-items:flex-start}
                .fee-table tbody td::before{margin-bottom:5px;flex:none}
                .fee-table tbody td .action-buttons-container{margin-left:0;width:100%;justify-content:flex-start}
                .fee-table tbody td .action-buttons-container .btn{font-size:0.75rem;padding:0.4rem 0.6rem}
                .quick-actions{grid-template-columns:1fr}
                .action-btn h4{font-size:14px}
                .action-btn p{font-size:12px}
                .modal-dialog{max-width:95%;margin:0.5rem auto}
                .modal-body{padding:1rem}
                .year-filter-container{margin-top:1rem;width:100%}
            }

            /* Extra small devices (portrait phones, less than 375px) */
            @media (max-width:375px){
                .page-content{padding:10px 5px}
                .summary-card{padding:15px}
                .summary-card .amount{font-size:22px}
                .fee-section{padding:15px 10px}
                .section-header h2{font-size:16px}
                .filter-btn{font-size:11px;padding:5px 10px;min-width:50px}
                .fee-table tbody tr{padding:0.75rem}
                .fee-table tbody td{font-size:0.8rem}
                .fee-table tbody td::before{font-size:0.75rem}
            }
</style>

{{-- JavaScript mantido igual --}}
<script>
(function () {
    'use strict';

    console.log('🔄 Script de pagamentos carregado - Versão 2.1');

    // =========================
    // Config
    // =========================
    const CONFIG = {
        PAYMENT_TIMEOUT: 5000,
        ANIMATION_DURATION: 300,
        VIBRATION_DURATION: 20,
        CURRENCY_LOCALE: 'pt-MZ',
        CURRENCY_OPTIONS: { minimumFractionDigits: 2, maximumFractionDigits: 2 }
    };

    // =========================
    // State + DOM Cache
    // =========================
    const state = {
        currentFeeData: {},
        currentReferenceData: {},
        isProcessingPayment: false,
        paymentTimeout: null
    };

    const elements = {};

    function initializeElements() {
        elements.modal = document.getElementById('collectFeeModal');
        elements.form = document.getElementById('paymentForm');
        elements.submitBtn = elements.form?.querySelector('button[type="submit"]');
        elements.paymentModeSelect = document.getElementById('paymentMode');

        // Modal display elements
        const ids = [
            'modalPeriod','modalFeeName','modalBaseAmount','modalCalculatedFine','modalTotalAmount',
            'amount','fine','hiddenMonth','hiddenYear','hiddenDiscount','summaryBase','summaryFine','summaryTotal'
        ];
        ids.forEach(id => elements[id] = document.getElementById(id));

        // Reference modal elements
        const refIds = ['refPeriod', 'refAmount', 'refExpires', 'refEntity', 'refReference'];
        refIds.forEach(id => elements[id] = document.getElementById(id));

        return true;
    }

    // =========================
    // Utils
    // =========================
    const utils = {
        formatCurrency(amount) {
            const number = parseFloat(amount) || 0;
            return new Intl.NumberFormat(CONFIG.CURRENCY_LOCALE, CONFIG.CURRENCY_OPTIONS).format(number) + ' MT';
        },
        parseNumber(v) { return parseFloat(v) || 0; },
    };

    // =========================
    // Modal: abrir/preencher
    // =========================
    window.openPaymentModal = function (month, year, baseAmount, calculatedFine, totalAmount, feeName, existingDiscount = 0) {
        state.currentFeeData = {
            month,
            year,
            baseAmount: utils.parseNumber(baseAmount),
            calculatedFine: utils.parseNumber(calculatedFine),
            totalAmount: utils.parseNumber(totalAmount),
            feeName: feeName || 'Taxa Escolar',
            existingDiscount: utils.parseNumber(existingDiscount)
        };

        console.log('Modal aberto com dados:', state.currentFeeData);

        state.isProcessingPayment = false;

        if (typeof $ !== 'undefined') $('#collectFeeModal').modal('show');

        setTimeout(() => {
            initializeElements();
            updateModalDisplay();

            // Registrar listener para mudança de método de pagamento
            if (elements.paymentModeSelect) {
                // Remover listeners antigos
                const newSelect = elements.paymentModeSelect.cloneNode(true);
                elements.paymentModeSelect.parentNode.replaceChild(newSelect, elements.paymentModeSelect);
                elements.paymentModeSelect = newSelect;

                elements.paymentModeSelect.addEventListener('change', function () {
                    console.log('Método de pagamento mudou para:', this.value);

                    const isReference = this.value === 'Reference' || this.value === 'Entity_Reference';

                    if (elements.submitBtn) {
                        elements.submitBtn.innerHTML = isReference
                            ? '<i class="fas fa-barcode me-1"></i>Gerar Referência'
                            : '<i class="fas fa-check me-1"></i>Confirmar Pagamento';
                    }

                    // Recalcular valores quando o método muda
                    updatePaymentSummary();
                });
            }
        }, 200);
    };

    // =========================
    // Reference Modal
    // =========================
    window.showReferenceModal = function (month, year, entity, reference, amount, expiresAt) {
        state.currentReferenceData = {
            month,
            year,
            entity,
            reference,
            amount,
            expiresAt
        };

        if (typeof $ !== 'undefined') $('#referenceModal').modal('show');

        setTimeout(() => {
            initializeElements();
            updateReferenceDisplay();
        }, 200);
    };

    function updateReferenceDisplay() {
        if (!state.currentReferenceData.month) return;

        if (!elements.refPeriod) initializeElements();

        elements.refPeriod && (elements.refPeriod.textContent = `${state.currentReferenceData.month} ${state.currentReferenceData.year}`);
        elements.refAmount && (elements.refAmount.textContent = state.currentReferenceData.amount);
        elements.refExpires && (elements.refExpires.textContent = state.currentReferenceData.expiresAt || 'Não definido');
        elements.refEntity && (elements.refEntity.textContent = state.currentReferenceData.entity || '-');
        elements.refReference && (elements.refReference.textContent = state.currentReferenceData.reference || '-');
    }

    window.copyReferenceData = function() {
        const entity = state.currentReferenceData.entity || '';
        const reference = state.currentReferenceData.reference || '';
        const amount = state.currentReferenceData.amount || '';
        
        const textToCopy = `Entidade: ${entity}\nReferência: ${reference}\nValor: ${amount}`;
        
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(textToCopy).then(() => {
                alert('Dados copiados para a área de transferência!');
            }).catch(() => {
                fallbackCopyTextToClipboard(textToCopy);
            });
        } else {
            fallbackCopyTextToClipboard(textToCopy);
        }
    };

    function fallbackCopyTextToClipboard(text) {
        const textArea = document.createElement("textarea");
        textArea.value = text;
        textArea.style.position = "fixed";
        textArea.style.left = "-999999px";
        textArea.style.top = "-999999px";
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        
        try {
            document.execCommand('copy');
            alert('Dados copiados para a área de transferência!');
        } catch (err) {
            console.error('Fallback: Could not copy text: ', err);
            alert('Não foi possível copiar os dados automaticamente.');
        }
        
        document.body.removeChild(textArea);
    }

    function updateModalDisplay() {
        if (!state.currentFeeData.month) return;

        if (!elements.modalPeriod) initializeElements();

        elements.modalPeriod && (elements.modalPeriod.textContent = `${state.currentFeeData.month} ${state.currentFeeData.year}`);
        elements.modalFeeName && (elements.modalFeeName.textContent = state.currentFeeData.feeName);
        elements.modalBaseAmount && (elements.modalBaseAmount.textContent = utils.formatCurrency(state.currentFeeData.baseAmount));
        elements.modalCalculatedFine && (elements.modalCalculatedFine.textContent = utils.formatCurrency(state.currentFeeData.calculatedFine));
        elements.modalTotalAmount && (elements.modalTotalAmount.textContent = utils.formatCurrency(state.currentFeeData.totalAmount));

        elements.amount && (elements.amount.value = state.currentFeeData.totalAmount.toFixed(2));
        elements.fine && (elements.fine.value = state.currentFeeData.calculatedFine.toFixed(2));
        elements.hiddenMonth && (elements.hiddenMonth.value = state.currentFeeData.month);
        elements.hiddenYear && (elements.hiddenYear.value = state.currentFeeData.year);
        elements.hiddenDiscount && (elements.hiddenDiscount.value = state.currentFeeData.existingDiscount.toFixed(2));

        updatePaymentSummary();
    }

    function updatePaymentSummary() {
        if (!state.currentFeeData.month) return;
        if (!elements.summaryBase) initializeElements();

        // Log para debug
        console.log('Atualizando resumo:', {
            base: state.currentFeeData.baseAmount,
            fine: state.currentFeeData.calculatedFine,
            total: state.currentFeeData.totalAmount,
            discount: state.currentFeeData.existingDiscount
        });

        elements.summaryBase && (elements.summaryBase.textContent = utils.formatCurrency(state.currentFeeData.baseAmount));
        elements.summaryFine && (elements.summaryFine.textContent  = utils.formatCurrency(state.currentFeeData.calculatedFine));
        elements.summaryTotal && (elements.summaryTotal.textContent = utils.formatCurrency(state.currentFeeData.totalAmount));
    }

    // =========================
    // Loading helpers
    // =========================
    function createLoadingOverlay() {
        const overlay = document.createElement('div');
        overlay.className = 'modal-loading-overlay';
        overlay.innerHTML = `
            <div class="modal-loading-spinner"></div>
            <p class="text-muted" data-overlay-text>Processando...</p>
        `;
        return overlay;
    }

    function showLoadingState(label = 'Processando...') {
        if (!elements.submitBtn) elements.submitBtn = document.querySelector('#paymentForm button[type="submit"]');
        if (elements.submitBtn) {
            const originalText = elements.submitBtn.innerHTML;
            elements.submitBtn.innerHTML = `<i class="fas fa-spinner fa-spin me-1"></i>${label}`;
            elements.submitBtn.disabled = true;
            elements.submitBtn.dataset.originalText = originalText;
        }

        const modalContent = document.querySelector('#collectFeeModal .modal-content');
        if (modalContent && !modalContent.querySelector('.modal-loading-overlay')) {
            modalContent.style.position = 'relative';
            modalContent.appendChild(createLoadingOverlay());
        }

        const txt = document.querySelector('.modal-loading-overlay [data-overlay-text]');
        if (txt) txt.textContent = label;

        setTimeout(() => {
            const overlay = document.querySelector('.modal-loading-overlay');
            if (overlay) overlay.classList.add('active');
        }, CONFIG.ANIMATION_DURATION);
    }

    function resetPaymentForm() {
        state.isProcessingPayment = false;

        if (elements.submitBtn) {
            elements.submitBtn.innerHTML = elements.submitBtn.dataset.originalText || '<i class="fas fa-check me-1"></i>Confirmar Pagamento';
            elements.submitBtn.disabled = false;
        }

        const overlay = document.querySelector('.modal-loading-overlay');
        if (overlay) overlay.classList.remove('active');
    }

    function hideModalAndShowSuccess() {
        state.isProcessingPayment = false;
        if (state.paymentTimeout) { clearTimeout(state.paymentTimeout); state.paymentTimeout = null; }

        try {
            if (typeof $ !== 'undefined' && $.fn.modal) {
                const $modal = $('#collectFeeModal');
                if ($modal.length && $modal.hasClass('show')) {
                    $modal.modal('hide');
                    setTimeout(() => {
                        document.querySelectorAll('.modal-backdrop').forEach(b => b?.parentNode?.removeChild(b));
                        document.body.classList.remove('modal-open');
                        document.body.style.removeProperty('padding-right');
                        document.body.style.removeProperty('overflow');
                    }, 500);
                }
            }
        } catch (e) {
            document.querySelectorAll('.modal-backdrop').forEach(el => el.remove());
            document.body.classList.remove('modal-open');
            document.body.style = '';
        }

        setTimeout(() => {
            alert('Pagamento efetuado com sucesso!');
            window.location.reload();
        }, CONFIG.ANIMATION_DURATION);
    }

    // =========================
    // Submit: fluxo normal vs referência
    // =========================
    window.submitPayment = function (event) {
        event.preventDefault();
        if (state.isProcessingPayment) return;

        state.isProcessingPayment = true;

        const formData = new FormData(event.target);
        const paymentData = Object.fromEntries(formData);

        if (!paymentData.paymentMode) {
            alert('Por favor, selecione um método de pagamento.');
            state.isProcessingPayment = false;
            return;
        }

        const isReference = paymentData.paymentMode === 'Reference' || paymentData.paymentMode === 'Entity_Reference';

        if (isReference) {
            showLoadingState('Gerando referência...');
            try {
                if (typeof Livewire !== 'undefined') {
                    Livewire.emit('generateReference', paymentData);
                } else {
                    throw new Error('Livewire not available');
                }
            } catch (error) {
                console.error('Reference error:', error);
                resetPaymentForm();
            }
            return;
        }

        showLoadingState('Processando...');
        state.paymentTimeout = setTimeout(() => {
            if (state.isProcessingPayment) hideModalAndShowSuccess();
        }, CONFIG.PAYMENT_TIMEOUT);

        try {
            if (typeof Livewire !== 'undefined') {
                Livewire.emit('collectPayment', paymentData);
            } else {
                throw new Error('Livewire not available');
            }
        } catch (error) {
            console.error('Payment error:', error);
            resetPaymentForm();
        }
    };

    // =========================
    // Filtro (lista)
    // =========================
    class FilterSystem {
        constructor() {
            this.filterButtons = null;
            this.tableContainer = null;
        }
        init() {
            setTimeout(() => {
                this.filterButtons = document.querySelectorAll('.filter-btn');
                this.tableContainer = document.querySelector('.fee-table-container');
                if (!this.tableContainer) return;
                this.setupFilters();
                this.applyFilter('all');
            }, 50);
        }
        setupFilters() {
            this.filterButtons.forEach(button => {
                button.addEventListener('click', (e) => {
                    e.preventDefault();
                    document.querySelectorAll('.filter-btn').forEach(btn => btn.classList.remove('active'));
                    button.classList.add('active');
                    const filterType = button.getAttribute('data-filter');
                    this.applyFilter(filterType);
                });
            });
        }
        applyFilter(filterType) {
            const tbody = this.tableContainer.querySelector('tbody');
            const rows = Array.from(tbody.querySelectorAll('tr[data-status]'));
            let visibleCount = 0;

            rows.forEach(row => {
                const status = row.getAttribute('data-status');
                const show = filterType === 'all' || status === filterType;
                if (show) {
                    row.style.display = '';
                    visibleCount++;
                } else {
                    row.style.display = 'none';
                }
            });

            this.handleNoResults(tbody, visibleCount);
        }
        handleNoResults(tbody, visibleCount) {
            let noResultsMsg = tbody.querySelector('.no-results-message');
            if (visibleCount === 0) {
                if (!noResultsMsg) {
                    noResultsMsg = document.createElement('tr');
                    noResultsMsg.className = 'no-results-message';
                    noResultsMsg.innerHTML = '<td colspan="100%" style="text-align:center; padding:20px;">Nenhum resultado encontrado para este filtro.</td>';
                }
                tbody.appendChild(noResultsMsg);
            } else if (noResultsMsg) {
                noResultsMsg.remove();
            }
        }
    }

    // =========================
    // DOM Ready
    // =========================
    document.addEventListener('DOMContentLoaded', () => {
        initializeElements();

        // Sistema de filtros
        const filterSystem = new FilterSystem();
        filterSystem.init();

        // Livewire integration
        if (typeof Livewire !== 'undefined') {
            document.addEventListener('livewire:load', () => {
                Livewire.on('paymentSuccess', () => hideModalAndShowSuccess());
                Livewire.on('paymentError', (message) => {
                    resetPaymentForm();
                    alert('Erro: ' + (message || 'Ocorreu um erro ao processar o pagamento.'));
                });

                Livewire.on('referenceReady', (payload) => {
                    resetPaymentForm();
                    if (typeof $ !== 'undefined') $('#collectFeeModal').modal('hide');
                    if (payload?.pdf_url) window.open(payload.pdf_url, '_blank');
                });

                Livewire.on('referenceError', (message) => {
                    resetPaymentForm();
                    alert(message || 'Não foi possível gerar a referência.');
                });

                Livewire.on('collectPayment', (data) => {
                    if (window.livewire?.components?.length > 0) {
                        const component = window.livewire.components[0];
                        component.set('month', data.month);
                        component.set('year', data.year);
                        component.set('amount', data.amount);
                        component.set('fine', data.fine);
                        component.set('discount', data.discount || 0);
                        component.set('paymentMode', data.paymentMode);
                        component.set('note', data.note || '');
                        component.call('collectFee');
                    }
                });
            });
        }
    });

    // =========================
    // Event Listeners para Comprovativo
    // =========================
    window.addEventListener('download-receipt', event => {
        const data = event.detail;

        if (data.url) {
            // Abrir em nova aba
            window.open(data.url, '_blank');

            // Mostrar mensagem de sucesso
            if (data.message) {
                showNotification('success', data.message);
            }
        }
    });

    window.addEventListener('error', event => {
        const data = event.detail;
        if (data.message) {
            showNotification('error', data.message);
        }
    });

    // Função para mostrar notificações
    function showNotification(type, message) {
        // Se existir Livewire Alert
        if (typeof window.Livewire !== 'undefined' && window.Livewire.emit) {
            const alertType = type === 'success' ? 'success' : 'error';
            window.Livewire.emit('alert', {
                type: alertType,
                message: message
            });
        } else {
            // Fallback para alert nativo
            alert(message);
        }
    }
})();
</script>

