<?php

namespace Chatify\Http\Controllers;

use App\Models\Admin;
use App\Models\ChFavorite as Favorite;
use App\Models\ChMessage as Message;
use App\Models\ChChannel as Channel;
use App\Models\User;
use App\Models\Classroom;
use App\Models\DeviceToken;
use Chatify\Facades\ChatifyMessenger as Chatify;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Exception\Messaging\NotFound;
use Kreait\Firebase\Factory;


class MessagesController extends Controller
{
    protected $perPage = 30;

    /**
     * Authenticate the connection for pusher
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function pusherAuth(Request $request)
    {
        return Chatify::pusherAuth(
            $request->user(),
            Auth::user(),
            $request['channel_name'],
            $request['socket_id']
        );
    }

    /**
     * Returning the view of the app with the required data.
     *
     * @param int $id
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    
    public function index($channel_id = null)
    {
        $messenger_color = Auth::user()->messenger_color;

        if(!Auth::user()->channel_id){
            Chatify::createPersonalChannel();
        }

        return view('Chatify::pages.app', [
            'channel_id' => $channel_id ?? 0,
            'channel' => $channel_id ? Channel::where('id', $channel_id)->first() : null,
            'messengerColor' => $messenger_color ? $messenger_color : Chatify::getFallbackColor(),
            'dark_mode' => Auth::user()->dark_mode < 1 ? 'light' : 'dark',
        ]);
    }
    /**
     * Fetch data (user, favorite.. etc).
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function idFetchData(Request $request)
    {
        $fetch = null;
        $channel_avatar = null;

        $favorite = Chatify::inFavorite($request['channel_id']);
        $channel = Channel::find($request['channel_id']);

        if(!$channel) return Response::json([
            'message' => "This chat channel doesn't exist!"
        ]);

        $allow_loading = $channel->owner_id === Auth::user()->id
            || in_array(Auth::user()->id, $channel->users()->pluck('id')->all());
       /* if(!$allow_loading) return Response::json([
             'message' => "You haven't joined this chat channel!"
        ]);*/

        // check if this channel is a group
        if(isset($channel->owner_id)){
            $fetch = $channel;
            $channel_avatar = Chatify::getChannelWithAvatar($channel)->avatar;
        } else {
            $fetch = Chatify::getUserInOneChannel($request['channel_id']);
            if($fetch){
                $channel_avatar = Chatify::getUserWithAvatar($fetch)->avatar;
            }
        }

        $infoHtml = view('Chatify::layouts.info', [
            'channel' => $channel,
        ])->render();

        return Response::json([
            'infoHtml' => $infoHtml,
            'favorite' => $favorite,
            'fetch' => $fetch ?? null,
            'channel_avatar' => $channel_avatar ?? null,
        ]);
    }

    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Symfony\Component\HttpFoundation\StreamedResponse|void
     */
    public function download($fileName)
    {
        $filePath = config('chatify.attachments.folder') . '/' . $fileName;
        if (Chatify::storage()->exists($filePath)) {
            return Chatify::storage()->download($filePath);
        }
        return abort(404, "Sorry, File does not exist in our server or may have been deleted!");
    }


    /** ORIGINAL METHOD
     * Send a message to database
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function send(Request $request)
    {
        // default variables
        $error = (object)[
            'status' => 0,
            'message' => null
        ];
        $attachment = null;
        $attachment_title = null;

        // if there is attachment [file]
        if ($request->hasFile('file')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();
            $allowed_files  = Chatify::getAllowedFiles();
            $allowed        = array_merge($allowed_images, $allowed_files);

            $file = $request->file('file');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed)) {
                    // get attachment name
                    $attachment_title = $file->getClientOriginalName();
                    // upload attachment and store the new name
                    $attachment = Str::uuid() . "." . $file->extension();
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
                } else {
                    $error->status = 1;
                    $error->message = "File extension not allowed!";
                }
            } else {
                $error->status = 1;
                $error->message = "File size you are trying to upload is too large!";
            }
        }

        if (!$error->status) {
            $lastMess = Message::where('to_channel_id', $request['channel_id'])->latest()->first();
             // send to database
             $message = Chatify::newMessage([
                'type' => $request['type'],
                'from_id' => Auth::user()->id,
                'to_channel_id' => $request['channel_id'],
                'body' => trim($request['message']),  // Removed htmlentities
                'attachment' => ($attachment) ? json_encode((object)[
                    'new_name' => $attachment,
                    'old_name' => trim($attachment_title),  // Removed htmlentities
                ]) : null,
            ]);
            $chanel = Channel::where('id', $request['channel_id'])->select('owner_id')->first();
            // load user info
            $message->user_avatar = Auth::user()->avatar;
            $message->user_name = Auth::user()->name;
            $message->user_email = Auth::user()->email;
            $message->owner_id=$chanel->owner_id;
            $messageData = Chatify::parseMessage($message, null, $lastMess ? $lastMess->from_id !== Auth::user()->id : true);

            Chatify::push("private-chatify.".$request['channel_id'], 'messaging', [
                'from_id' => Auth::user()->id,
                'to_channel_id' => $request['channel_id'],
                'message' => $messageData 
                //Chatify::messageCard($messageData, true)
            ]);
              // Send notification after successfully saving the message
              $this->sendNotification($request['channel_id'], $message, $attachment);
        }
       
        // send the response
        return Response::json([
            'status' => '200',
            'error' => $error,
            'message' => Chatify::messageCard(@$messageData),
            'tempID' => $request['temporaryMsgId'],
        ]);
    }



// /**
//  * Send a message to database (ALL IN SAME FUNCTION... SEND NOTIFICATION BUT MESSAGE DISAPEAR)
//  *
//  * @param Request $request
//  * @return JsonResponse
//  */
// public function send(Request $request)
// {
//     // Default error object
//     $error = (object)[
//         'status' => 0,
//         'message' => null
//     ];

//     // Process attachments (if any)
//     $attachment = null;
//     $attachment_title = null;
    
//     //if there is attachment [file]
//     if ($request->hasFile('file')) {
//         // allowed extensions
//         $allowed_images = Chatify::getAllowedImages();
//         $allowed_files  = Chatify::getAllowedFiles();
//         $allowed = array_merge($allowed_images, $allowed_files);

//         $file = $request->file('file');
//         // check file size
//         if ($file->getSize() < Chatify::getMaxUploadSize()) {
//             if (in_array(strtolower($file->extension()), $allowed)) {
//                 // get attachment name
//                 $attachment_title = $file->getClientOriginalName();
//                 // upload attachment and store the new name
//                 $attachment = Str::uuid() . "." . $file->extension();
//                 $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
//             } else {
//                 $error->status = 1;
//                 $error->message = "File extension not allowed!";
//             }
//         } else {
//             $error->status = 1;
//             $error->message = "File size you are trying to upload is too large!";
//         }
//     }

//     // Log request details
//     Log::info('Starting message send process for channel_id: ' . $request['channel_id']);

//     if (!$error->status) {
//         // Send message to database
//         try {
//             $lastMess = Message::where('to_channel_id', $request['channel_id'])->latest()->first();
//             $message = Chatify::newMessage([
//                 'type' => $request['type'],
//                 'from_id' => Auth::user()->id,
//                 'to_channel_id' => $request['channel_id'],
//                 'body' => trim($request['message']),
//                 'attachment' => $attachment ? json_encode((object)[
//                     'new_name' => $attachment,
//                     'old_name' => trim($attachment_title),
//                 ]) : null,
//             ]);

//             Log::info('Message saved to database: ', ['message_id' => $message->id]);
//         } catch (\Exception $e) {
//             Log::error('Error saving message to database: ' . $e->getMessage());
//         }

//         // Retrieve user IDs associated with the channel from the ch_channel_user table
//         try {
//             $userIds = DB::table('ch_channel_user')
//                 ->where('channel_id', $request['channel_id'])
//                 ->pluck('user_id');

//             if ($userIds->isEmpty()) {
//                 throw new \Exception("No users found for channel_id: " . $request['channel_id']);
//             }
//             Log::info('User IDs for channel: ' . json_encode($userIds));

//         } catch (\Exception $e) {
//             Log::error('Error retrieving users for channel: ' . $e->getMessage());
//             return response()->json([
//                 'status' => '404',
//                 'error' => 'No users found for channel_id',
//                 'message' => $e->getMessage(),
//             ], 404);
//         }
//     }

//     // Sending Firebase Notifications
//     try {
//         $firebaseCredentials = [
//         "type" => "service_account",
//         "project_id" => "copmoz-3x",
//         "private_key_id" => "7768bcef3df838b966f6ec0f3fd398ced9336439",
//         "private_key" => "-----BEGIN PRIVATE KEY-----\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDY12nvQbs5PGiU\nuZrrhOKpCB5viTy6PdNNB0vz8MRA+8SL4RaZG8b/AA9qgNRj03GMx3cP6OVFMi7C\n7/KRjjHJisfK0etOqfRc8IVrG4pizwmetMLdE1PsSNsHaC/mddLQKbKwsf8QhJGb\npSRI4PMdgSGmsvpn0L0ki7DhrdRF7d8NdFFLdrXTtULSmpWyh+Mvt9GApM/ffQh6\nAC0X04BANj449xCAJVFsjfDrXgN4/ZOOBUItLZK0ICDjuaf8009ABdxn28Y0oRQC\nIk4fbPtBcLhofFyXiGDV0XiAvgCioyM4eTPDLfU3d9NbgF7pHo2aust4qg2qt5/k\ncVKN4TMhAgMBAAECggEAMOaXdcTpj3VPkmavCLkNkpgJoRaectO5gLAKEWe1IoCE\nqRToTzwD6wLc9yQ5gY+wFlLvR0+60JmKER0io2+hS7W368XeXn2WZZU6SVbLJRBS\nt49FOK0AqpMdjfSRhNQCaD5XFWD2yAdVfPfHkIXmXbt+1sSbEulwljrId4SHkAMz\n+epUKT48svL+Kbdijz9zZjzw1LkX768tn4G+7vDNv2VgQnZbWHun0beOHvYesb7A\ncHQqT20B8wHRlVzUV5Mj61Tzgn84Oq+7TFwFkw/kYu3ErtP+Cwv+LWExnyeZ2ZMe\nvGPQRvN9BRbr5gyIJzc0FX9jPzfI/7lbdePVdgHNTwKBgQD3i9aSfBO+SiMsh3y8\nsBkE8sG6bP/lXk9ySLEV/3LFB9sRJpFOe4hCOvFY2wb0pHbVZd65o2FdmvRZm5DK\nrJA8oqByOIt9x9dBMIEQW4IrNNOeJwEflpagL/32KH4L16PPcf3CYZRLQCxVRMlm\n+FTTtGDpydnA/e8YO8weGU2aDwKBgQDgPyP1QiIbvotj9V3hNXYfiT88m/Al9roe\nHkQ2psE3YCz4BQGQ8xfPYmY3dDyHFegPLHHDiHMkvPTcFxt0S9P7k/eKOLBpgp9e\nihrNoDXBD5B0wgLQc1Mp4HYEeTnH9mEKOGURznOIF8X4uVY7+WzV56Rb38NZHIi7\nf0J54/lPzwKBgGbm92RfNanVkbeR/rVFErFu5D9Fzj4frTDZrNtzG9dxw4vml2Kj\nkWnOrML+EW8i6s8ckLBkjTXnEwoo09CcWT0LcVzpynqX1xAfRJdzIrIpPIqythHW\nfHjMWmq4PGVZ15uFTDJAc++wFn2/oWUvd7ulcN/ea6mBJaHSQbCFXi/bAoGBALu1\nvtjPShPlVXMP8kzRXpaZYXjPo59zv2n0te71eWlsNVhLgHsgMf8kBCJJR11ghgcl\nkfk2YAQRs/oD0bRwwPqnap+mQm2frN4LRtJ3WGfKKobB3G83ChdU0BWAkFCFOce/\nW0MjiOwT0mEYa0n6yRLpO1i4nyIZ0473wOKdZZVbAoGBAL4+ffKdzPJy4SG4v0L5\nPvYPW4u2gukA3G4aTWOg2EwUGq2HLaGFCvDqE+hR3bGqVEhk6X12vHuke4UFaN44\nXUfs8v65F/LT7VG0+dhfpDzbfhnVoTqsJOB3obQnQzJPfk4FThEQOBqepKV8PDyo\nVosJTLofmtwZH823UAZIIOd/\n-----END PRIVATE KEY-----\n",
//         "client_email" => "firebase-adminsdk-as4sm@copmoz-3x.iam.gserviceaccount.com",
//         "client_id" => "112030584520787629755",
//         "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
//         "token_uri" => "https://oauth2.googleapis.com/token",
//         "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
//         "client_x509_cert_url" => "https://www.googleapis.com/robot/v1/metadata/x509/firebase-adminsdk-as4sm%40copmoz-3x.iam.gserviceaccount.com",
//         "universe_domain" => "googleapis.com"
//         ];

//         $factory = (new Factory)->withServiceAccount($firebaseCredentials);
//         $messaging = $factory->createMessaging();
//         Log::info('Firebase Messaging initialized');

//         // For each user in the channel, send a notification
//         foreach ($userIds as $userId) {
//             $user = User::where('id', $userId)->first();
//             if (!$user) {
//                 Log::error("User not found for ID: $userId");
//                 continue;
//             }

//             // Get device tokens for the user
//             $tokens = DeviceToken::where('user_id', $user->id)->pluck('token');
//             if ($tokens->isEmpty()) {
//                 Log::error("No device tokens found for user: " . $user->name);
//                 continue;
//             }

//             Log::info('Device tokens retrieved for user: ' . $user->name, ['tokens' => $tokens]);

//             // Send notification for each device token
//             foreach ($tokens as $token) {
//                 try {
//                     $notification = Notification::fromArray([
//                         'token' => $token,
//                         'title' => "Nova mensagem de " . Auth::user()->name,
//                         'body' => ($attachment) ? "Ficheiro anexado" : $message->body,
//                     ]);

//                     $firebaseMessage = CloudMessage::withTarget('token', $token)
//                         ->withNotification($notification)
//                         ->withHighestPossiblePriority()
//                         ->withDefaultSounds();

//                     $messaging->send($firebaseMessage);
//                     Log::info('Notification sent to token: ' . $token);

//                 } catch (\Kreait\Firebase\Exception\Messaging\InvalidMessage $e) {
//                     Log::error("Invalid token: {$token}. Error: {$e->getMessage()}");
//                     DeviceToken::where('token', $token)->delete();  // Remove invalid token
//                 } catch (\Exception $e) {
//                     Log::error("Error sending notification: " . $e->getMessage());
//                 }
//             }
//         }

//     } catch (\Exception $e) {
//         Log::error('Error initializing Firebase or sending notifications: ' . $e->getMessage());
//         return response()->json([
//            'status' => '500',
//             'error' => 'Internal Server Error',
//             'message' => 'Failed to send notifications',
//         ], 500);
//     }

//     // Final response
//     return response()->json([
//         'status' => '200',
//         'error' => $error,
//         'message' => Chatify::messageCard(@$messageData),
//         'tempID' => $request['temporaryMsgId'],
//     ]);
// }

protected function sendNotification($channelId, $message, $attachment)
{
    try {
        // Retrieve user IDs associated with the channel from the ch_channel_user table
        $userIds = DB::table('ch_channel_user')
            ->where('channel_id', $channelId)
            ->pluck('user_id');

        if ($userIds->isEmpty()) {
            throw new \Exception("No users found for channel_id: " . $channelId);
        }

        $firebaseCredentials = [
            "type" => "service_account",
            "project_id" => "copmoz-3x",
            "private_key_id" => "bb9e045f0e2345c069b299a14636203509899ef8",
            "private_key" => "-----BEGIN PRIVATE KEY-----\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDYfKrqJJyR6/8w\nP0NI75wA0OMzBMq53JD0GX/2G488VD2gE+Wbxt90e/6PUGdrOEwo7C+ilCNKsoX7\nzyy6l6kOt6MgVjO39bc7/qwaKYcA05/48grMrLMUxb6SNAMmKmj3CTQ9dU/zAmqA\n96HleqAkHt22wRSQ8hy4qg44CC1f8faBJxGb68YeMm9TmoYcdUhc7QT13JoMRZjl\nbi5gNDVIFWDyUhXeVpvw13Z3Sfo+EgkiT2Tb2Nr8//rptIabmrhzUsAFTEKnHblh\nY+1nsxBBwImu1NRtr5aQUCNLUZXcbWpuoym6EZtCLe3u6/mL3wI05407q1YqpUOq\nppfDwuHdAgMBAAECggEAMuv0Kfotp/dKgddHpObTm6+G2a/VXa2fnQNIgaPO5+Jq\n0nYLqr43PoeSA7TmdqZ556BSvhu5R29Bq876FvPojLEuWkNci/dsv9xyL/83pb6h\noOWSG729q/OtlWdNnFMaeI0+L/yhrygQ1EArYTacZ5KpFT1KcKO6lMdPa+ekgK4T\nPuwGVD/3VDFG9AWCEm+v3w6bQAhpKbdZiBPpGvTrfFHYov97/Yzv/iC3599VnW+i\no3GweFGaqsOn4PxuyIyYujzgoIc6kBrivUV+ddxZqLufrsQL1uE78SuWjhRNS8/k\n5rUD3Qvaj3BOQzkqCnttclfPZYSmbjMtVyLOMjMcQQKBgQDkE1SnrhY/dZF5QCFg\nIMaf7wkPLLCSStugOeYMGx2oeD0MxaY9L9N1ffRLekngffzANrO/ybE7AlRzgBJB\n2NYFd7fueihmrWuTpMafkBivG+mf+HdTjkxF0aQfYUL+MMW7viBBzGDbwckRSTX5\n5pSqzEHmRvctiLTfHqqgUnDeIQKBgQDy/hzbdteqvgqRok7KoXy3ciij4SQbDOQZ\n19KcmMTFgMiT/xm/3uni924Qw82VVSYQjDbAoSltTl7cpvtkXzSB2LIg00YEhfcE\nj/atYfswDIBYBAGCUZDalOn6s/8TWuTGVfCkQVwHA2Xa3rKqIrI1gHTuu5N4J/P4\niCmL3Vd0PQKBgQC1mirC5Rvz5ZIywHySxSZCfJtzCLuDkLV1zAQ/yWuBFFRQEiS+\n/ZDbMbJOUw32AXs6NZREdr125fEGkoh3A8fOTLgY3A3FS/qncgFxVdRBwfDxHm8t\nCdXzleyfy9sC+STIy6d1nN3WvWNzLx8aX54qrT8fs4vnJr4WFp330Azs4QKBgQCc\nnDj/Hb5sdmWbbW+425HlLfeRf+bkZE+TdG1yrmeMH6+m6zCuYD6AIRbYSUp0J9gL\nrEMRrg1kPLGZJyo5i77svTw5OdIT0j/dueez4cWiNzx0/cf3NRjWOEoBYgdRczCl\nv/gD9XgZsQ/xm7ytOQWAxBUZVN38AEwW77NPSM43xQKBgCgCmFCO57u8Fc6d+4G9\nBy/Vfmy3tZpwGCTGqxkwhFvG+qXZTERJOwRg4jejIM/W6iQlOPmfsis2CWhCZvYL\nSimfvuVeWSVixdnsxVpjVyyT7ci9jMhPXeL7LtMCjNF4hB34qgejqpkwh62Zqg0h\nNuy5ZdoARi0VwY/spf6HK03b\n-----END PRIVATE KEY-----\n",
            "client_email" => "firebase-adminsdk-as4sm@copmoz-3x.iam.gserviceaccount.com",
            "client_id" => "112030584520787629755",
            "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
            "token_uri" => "https://oauth2.googleapis.com/token",
            "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
            "client_x509_cert_url" => "https://www.googleapis.com/robot/v1/metadata/x509/firebase-adminsdk-as4sm%40copmoz-3x.iam.gserviceaccount.com",
            "universe_domain" => "googleapis.com"
        ];

        $factory = (new Factory)->withServiceAccount($firebaseCredentials);
        $messaging = $factory->createMessaging();
        Log::info('Firebase Messaging initialized');

        // For each user in the channel, send a notification
        foreach ($userIds as $userId) {
            $user = User::where('id', $userId)->first();
            if (!$user) {
                Log::error("User not found for ID: $userId");
                continue;
            }

            // Get device tokens for the user
            $tokens = DeviceToken::where('user_id', $user->id)->pluck('token');
            if ($tokens->isEmpty()) {
                Log::error("No device tokens found for user: " . $user->name);
                continue;
            }

            Log::info('Device tokens retrieved for user: ' . $user->name, ['tokens' => $tokens]);

            // Send notification for each device token
            foreach ($tokens as $token) {
                try {
                    $notification = Notification::fromArray([
                        'token' => $token,
                        'title' => "Nova mensagem de " . Auth::user()->name,
                        'body' => ($attachment) ? "Ficheiro anexado" : $message->body,
                    ]);

                    $firebaseMessage = CloudMessage::withTarget('token', $token)
                        ->withNotification($notification)
                        ->withHighestPossiblePriority()
                        ->withDefaultSounds();

                    $messaging->send($firebaseMessage);
                    Log::info('Notification sent to token: ' . $token);

                } catch (\Kreait\Firebase\Exception\Messaging\InvalidMessage $e) {
                    Log::error("Invalid token: {$token}. Error: {$e->getMessage()}");
                    DeviceToken::where('token', $token)->delete();  // Remove invalid token
                } catch (\Exception $e) {
                    Log::error("Error sending notification: " . $e->getMessage());
                }
            }
        }

    } catch (\Exception $e) {
        Log::error('Error sending notifications: ' . $e->getMessage());
    }
}




    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function fetch(Request $request)
    {
        $query = Chatify::fetchMessagesQuery($request['id'])->latest();
        $messages = $query->paginate($request->per_page ?? $this->perPage);
        $totalMessages = $messages->total();
        $lastPage = $messages->lastPage();
        $response = [
            'total' => $totalMessages,
            'last_page' => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages' => '',
        ];

        // if there is no messages yet.
        if ($totalMessages < 1) {
            $response['messages'] ='<p class="message-hint center-el"><span>Diga \'Ola\' para iniciar a conversa</span></p>';
            return Response::json($response);
        }
        if (count($messages->items()) < 1) {
            $response['messages'] = '';
            return Response::json($response);
        }
        $allMessages = null;
        $prevMess = null;
        foreach ($messages->reverse() as $message) {
            $allMessages .= Chatify::messageCard(
                Chatify::parseMessage($message, null, $prevMess ? $prevMess->from_id != $message->from_id : false)
            );
            $prevMess = $message;
        }
        $response['messages'] = $allMessages;
        return Response::json($response);
    }

    /**
     * Make messages as seen
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function seen(Request $request)
    {
        try {
            $userId = Auth::user()->id;
            // Aceitar tanto 'id' quanto 'channel_id' (JS envia 'channel_id')
            $channelId = $request['channel_id'] ?? $request['id'];

            Log::info('📩 [Chatify Web] Marcando mensagens como vistas', [
                'user_id' => $userId,
                'channel_or_user_id' => $channelId,
                'request_data' => $request->all(),
            ]);

            // Tentar marcar como visto usando canal primeiro
            $updatedCount = Message::where('to_channel_id', $channelId)
                ->where('from_id', '!=', $userId) // Não marcar mensagens próprias
                ->where('seen', 0)
                ->update([
                    'seen' => 1,
                    'updated_at' => now()
                ]);

            // Se não atualizou nada, tentar método antigo (conversas diretas sem canal)
            if ($updatedCount === 0) {
                $updatedCount = Message::where('from_id', $channelId)
                    ->where('to_id', $userId)
                    ->where('seen', 0)
                    ->update([
                        'seen' => 1,
                        'updated_at' => now()
                    ]);
            }

            Log::info('✅ [Chatify Web] Mensagens marcadas como vistas', [
                'channel_or_user_id' => $channelId,
                'updated_count' => $updatedCount,
            ]);

            return Response::json([
                'status' => 1,
                'updated_count' => $updatedCount,
            ], 200);

        } catch (\Exception $e) {
            Log::error('❌ [Chatify Web] Erro ao marcar mensagens como vistas: ' . $e->getMessage());
            return Response::json([
                'status' => 0,
                'error' => $e->getMessage(),
            ], 500);
        }
    }

	
    /**
     * Get contacts list
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getContacts(Request $request)
    {
        $query = Channel::join('ch_messages', 'ch_channels.id', '=', 'ch_messages.to_channel_id')
            ->join('ch_channel_user', 'ch_channels.id', '=', 'ch_channel_user.channel_id')
            ->where('ch_channel_user.user_id','=',Auth::user()->id)
            ->select('ch_channels.*', DB::raw('ch_messages.created_at messaged_at'))
            ->groupBy('ch_channels.id')
            ->orderBy('messaged_at', 'desc')
            ->paginate($request->per_page ?? $this->perPage);

        $channelsList = $query->items();

        if (count($channelsList) > 0) {
            $contacts = '';
            foreach ($channelsList as $channel) {
                $contacts .= Chatify::getContactItem($channel);
            }
        } else {
            $contacts = '<p class="message-hint center-el"><span>Sua lista de contactos está vazia!</span></p>';
        }

        return Response::json([
            'contacts' => $contacts,
            'total' => $query->total() ?? 0,
            'last_page' => $query->lastPage() ?? 1,
        ], 200);
    }

    /**
     * Update user's list item data
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function updateContactItem(Request $request)
    {
		$channel_id = $request['channel_id'];
		
        // Get user data
        $channel = Channel::find($channel_id);
        if(!$channel){
            return Response::json([
                'message' => 'Channel not found!',
            ], 401);
        }
        $contactItem = Chatify::getContactItem($channel);

        // send the response
        return Response::json([
            'contactItem' => $contactItem,
        ], 200);
    }
    public function getChannelId(Request $request)
	{
		$user_id = $request['user_id'];
		$res = Chatify::getOrCreateChannel($user_id);
		
		// send the response
		return Response::json($res, 200);
	}

    /**
     * Put a user in the favorites list
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function favorite(Request $request)
    {
        $channel_id = $request['channel_id'];
        // check action [star/unstar]
        $favoriteStatus = Chatify::inFavorite($channel_id) ? 0 : 1;
        Chatify::makeInFavorite($channel_id, $favoriteStatus);

        // send the response
        return Response::json([
            'status' => @$favoriteStatus,
        ], 200);
    }

    /**
     * Get favorites list
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function getFavorites(Request $request)
    {
        $favoritesList = null;
        $favorites = Favorite::where('user_id', Auth::user()->id);
        foreach ($favorites->get() as $favorite) {
            $channel = Channel::find($favorite->favorite_id);
            $user=false;
            $data = null;
            if($channel->owner_id){
                $data = Chatify::getChannelWithAvatar($channel);
            } else {
                $user = Chatify::getUserInOneChannel($channel->id);
                $data = Chatify::getUserWithAvatar($user);
            }
            $favoritesList .= view('Chatify::layouts.favorite', [
                'data' => $data,
                'user' => $user,
                'channel_id' => $channel->id
            ]);
        }
        // send the response
        return Response::json([
            'count' => $favorites->count(),
            'favorites' => $favorites->count() > 0
                ? $favoritesList
                : 0,
        ], 200);
    }

    /**
     * Search in messenger
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function search(Request $request)
    { 
        $getRecords = null;
        $input = trim(filter_var($request['input']));
        $records = User::select("id", "name", "avatar","active_status",  DB::raw("-1 as role"))->where('id', '!=', Auth::user()->id)
            ->where('name', 'LIKE', "%{$input}%");
        $recordsteacher = Admin::select("id", "name", "avatar", "active_status", "role" )->where('id', '!=', Auth::user()->id)
            ->where('name', 'LIKE', "%{$input}%")
            ->union($records)
            ->paginate($request->per_page ?? $this->perPage);

        foreach ($recordsteacher->items() as $record) {
            $getRecords .= view('Chatify::layouts.listItem', [
                'get' => 'search_item',
                'user' => Chatify::getUserWithAvatar($record),
            ])->render();
        }

        if ($recordsteacher->total() < 1) {
            $getRecords = '<p class="message-hint center-el"><span>Nada para mostrar.</span></p>';
        }
        // send the response
        return Response::json([
            'records' => $getRecords,
            'total' => $recordsteacher->total(),
            'last_page' => $recordsteacher->lastPage(),
        ], 200);
    }

    /**
     * Get shared photos
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function sharedPhotos(Request $request)
    {
        $shared = Chatify::getSharedPhotos($request['channel_id']);
        $sharedPhotos = null;

        // shared with its template
        for ($i = 0; $i < count($shared); $i++) {
            $sharedPhotos .= view('Chatify::layouts.listItem', [
                'get' => 'sharedPhoto',
                'image' => Chatify::getAttachmentUrl($shared[$i]),
            ])->render();
        }
        // send the response
        return Response::json([
            'shared' => count($shared) > 0 ? $sharedPhotos : '<p class="message-hint"><span>Nada partilhado</span></p>',
        ], 200);
    }

    /**
     * Delete conversation
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteConversation(Request $request)
    {
        // delete
        $delete = Chatify::deleteConversation($request['channel_id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function deleteGroupChat(Request $request)
    {
        $channel_id = $request['channel_id'];


        $channel = Channel::findOrFail($channel_id);
        $channel->users()->detach();

        Chatify::deleteConversation($channel_id);


        // send the response
        return Response::json([
            'deleted' => $channel->delete(),
        ], 200);
    }
    public function leaveGroupChat(Request $request)
    {
        $channel_id = $request['channel_id'];
        $user_id = $request['user_id'];

        // add last message
        $message = Chatify::newMessage([
            'from_id' => Auth::user()->id,
            'to_channel_id' => $channel_id,
            'body' => Auth::user()->name . ' saiu do grupo',
            'attachment' => null,
        ]);
        $message->user_avatar = Auth::user()->avatar;
        $message->user_name = Auth::user()->name;
        $message->user_email = Auth::user()->email;

        $messageData = Chatify::parseMessage($message, null);

        Chatify::push("private-chatify.".$channel_id, 'messaging', [
            'from_id' => Auth::user()->id,
            'to_channel_id' => $channel_id,
            'message' => Chatify::messageCard($messageData, true)
        ]);

        // detach user
        $channel = Channel::findOrFail($channel_id);
        $channel->users()->detach($user_id);

        // send the response
        return Response::json([
            'left' => $channel ? 1 : 0,
        ], 200);
    }

    /**
     * Delete message
     *
     * @param Request $request
     * @return JsonResponse
     */


    public function deleteMessage(Request $request)
    {
        // delete
        $delete = Chatify::deleteMessage($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function updateSettings(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        // dark mode
        if ($request['dark_mode']) {
            $request['dark_mode'] == "dark"
            ? User::where('id', Auth::user()->id)->update(['dark_mode' => 1]) // Make Dark
            : User::where('id', Auth::user()->id)->update(['dark_mode' => 0]); // Make Light
        }

        // If messenger color selected
        if ($request['messengerColor']) {
            $messenger_color = trim(filter_var($request['messengerColor']));
            User::where('id', Auth::user()->id)
                ->update(['messenger_color' => $messenger_color]);
        }
     
       
        // if there is a [file]
        if ($request->hasFile('avatar')) {
           
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();
           
            $file = $request->file('avatar');
            
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
               
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    // delete the older one
                    if (Auth::user()->avatar != config('chatify.user_avatar.default')) {
                        $avatar = Auth::user()->avatar;
                        if ($avatar!=null&&Chatify::storage()->exists($avatar)) {
                            Chatify::storage()->delete($avatar);
                        }
                    }
                    // upload
 
                    $avatar = Str::uuid() . "." . $file->extension();
                    if(Auth::guard('admin')->check()){
                        $update = Admin::where('id', Auth::user()->id)->update(['avatar' => Chatify::storage()->url(config('chatify.user_avatar.folder') . '/' . $avatar)]);
                    }
                    else  $update = User::where('id', Auth::user()->id)->update(['avatar' => Chatify::storage()->url(config('chatify.user_avatar.folder') . '/' . $avatar)]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        // send the response
        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
        ], 200);
    }

    /**
     * Set user's active status
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function setActiveStatus(Request $request)
    {
        $activeStatus = $request['status'] > 0 ? 1 : 0;
        $status = User::where('id', Auth::user()->id)->update(['active_status' => $activeStatus]);
        $status = Admin::where('id', Auth::user()->id)->update(['active_status' => $activeStatus]);
        return Response::json([
            'status' => $status,
        ], 200);
    }

    public function searchUsers(Request $request)
    {
        $Tumas=false;
        $sTurmas=false;
        $Classrooms=false;
        $getRecords = array();
        $input = trim(filter_var($request['input']));
        $searchclass =  $request['searchclass'];
        $searchTurmas =  $request['searchTurmas'];
        if(!empty($searchclass)){
           $Classrooms= Classroom::select("id")->where('class',$searchclass )->get();
           $Turmas= Classroom::select("name")->where('class',$searchclass )->groupBy('name' )->get();
        
        if(!empty($searchTurmas)){
            
            $sTurmas= Classroom::select("id")->where('name',$searchTurmas )->whereIn('id',  $Classrooms)->get();
         }
        }
            $records = User::select("id", "name", "avatar", "active_status", DB::raw("-1 as role"))->where('id', '!=', Auth::user()->id)
            ->where('name', 'LIKE', "%{$input}%");
            if($sTurmas){ 
                $records  ->whereIn('classroom_id',  $sTurmas);
            }
            else
            if($Classrooms){
                $records  ->whereIn('classroom_id',  $Classrooms);
            }
            if(!$Classrooms&&!$sTurmas){
                $recordsteacher = Admin::select("id", "name", "avatar","active_status", "role" )->where('id', '!=', Auth::user()->id)
                ->where('name', 'LIKE', "%{$input}%")
                ->union($records)
                ->paginate(50);
            }
            else{
                $recordsteacher=$records->paginate(50);
            }
      

        foreach ($recordsteacher->items() as $record) {
            $getRecords[] = array(
                "user" => $record,
                "view" => view('Chatify::layouts.listItem', [
                    'get' => 'user_search_item',
                    'user' => Chatify::getUserWithAvatar($record),
                ])->render()
            );
        }
        if($recordsteacher->total() < 1){
            $getRecords = '<p class="message-hint"><span>Nada para mostrar.</span></p>';
        }
        // send the response
        return Response::json([
            'records' => $getRecords,
            
            'sTurmas' => $sTurmas,
            'turmas' => $Turmas,
            'total' => $recordsteacher->total(),
            'last_page' => $recordsteacher->lastPage()
        ], 200);
    }


    
    public function createGroupChat(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        $user_ids =  explode(',', $request['user_ids']);
        $user_ids[] = Auth::user()->id;

        $group_name = $request['group_name'];

        $new_channel = new Channel();
        $new_channel->name = $group_name;
        $new_channel->owner_id = Auth::user()->id;
        $new_channel->save();
        $new_channel->users()->sync($user_ids);
        $new_channel->admins()->sync($user_ids);
        // add first message
        $message = Chatify::newMessage([
            'from_id' => Auth::user()->id,
            'to_channel_id' => $new_channel->id,
            'body' => Auth::user()->name . ' criou um novo grupo: ' . $group_name,
            'attachment' => null,
        ]);
        $message->user_name = Auth::user()->name;
        $message->user_email = Auth::user()->email;

        $messageData = Chatify::parseMessage($message, null);
        Chatify::push("private-chatify.".$new_channel->id, 'messaging', [
            'from_id' => Auth::user()->id,
            'to_channel_id' => $new_channel->id,
            'message' => Chatify::messageCard($messageData, true)
        ]);


        // if there is a [file]
        if ($request->hasFile('avatar')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();

            $file = $request->file('avatar');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    $avatar = Str::uuid() . "." . $file->extension();
                    $update = $new_channel->update(['avatar' => Chatify::storage()->url(config('chatify.user_avatar.folder') . '/' . $avatar)]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
            'channel' => $new_channel
        ], 200);
    }
}
