<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\Fee_assign;
use App\Models\User;
use App\Services\FeeCalculationService;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use App\Models\FeesDiscount as discount;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class FeeCollection extends Component
{
    public $studentId;
    public $feeGroup;
    public $date;
    public $amount;
    public $discountGroup;
    public $discount = 0;
    public $fine = 0;
    public $paymentMode = 'Cash';
    public $note;
    public $students = [];
    public $selectedStudent;
    public $selectedFee;
    public $selectedFeeAssignId;
    public $month;
    public $year;

    // Cálculos atuais (form do modal)
    public $calculatedBaseFee = 0;
    public $calculatedFine = 0;
    public $calculatedTotal = 0;
    public $isLatePayment = false;
    public $dueDate = null;

    // Filtro/Export
    public $selectedYear;

    protected $feeCalculationService;

    /* -------------------------- Boot / Mount -------------------------- */

    public function boot(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount($studentId)
    {
        $this->studentId = $studentId ?: "cpmc-1218";
        $this->selectedStudent = User::findOrFail($studentId);
        $this->date = now()->format('d/m/Y');
        $this->selectedYear = now()->year;

        $this->calculateCurrentFee();
    }

    public function updated($propertyName)
    {
        if (in_array($propertyName, ['month', 'year', 'discount', 'fine'])) {
            $this->calculateCurrentFee();
        }
    }

    /* ------------------------ Cálculo em tempo real ------------------------ */

    public function calculateCurrentFee()
    {
        if (!$this->selectedStudent) return;

        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $this->month,
                $this->year ? (int) $this->year : null
            );

            $this->calculatedBaseFee = (float) $calculation['base_amount'];
            $this->calculatedFine    = (float) $calculation['fine_amount'] + (float) $this->fine; // auto + manual
            $this->calculatedTotal   = $this->calculatedBaseFee + $this->calculatedFine - (float) $this->discount;
            $this->isLatePayment     = (bool) $calculation['is_late_payment'];
            $this->dueDate           = $calculation['due_date'];

            // O campo "amount" no DB é LÍQUIDO (base + multa - desconto)
            $this->amount = max(0, $this->calculatedTotal);

            $this->emit('feeCalculated', [
                'baseFee'  => $this->calculatedBaseFee,
                'autoFine' => (float) $calculation['fine_amount'],
                'manualFine' => (float) $this->fine,
                'totalFine'  => $this->calculatedFine,
                'total'      => $this->calculatedTotal,
                'isLate'     => $this->isLatePayment,
                'dueDate'    => $this->dueDate?->format('d/m/Y'),
            ]);
        } catch (\Exception $e) {
            \Log::error('Error calculating fee', [
                'student_id' => $this->studentId,
                'month' => $this->month,
                'year' => $this->year,
                'error' => $e->getMessage(),
            ]);
            $this->fallbackCalculation();
        }
    }

    private function fallbackCalculation()
    {
        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        if (!$studentClassroom) {
            $this->calculatedBaseFee = 0;
            $this->calculatedFine = (float) $this->fine;
            $this->calculatedTotal = (float) $this->fine - (float) $this->discount;
            return;
        }

        $feeStructures = FeeStructure::where('active', 1)->get();
        $filtered = [];

        foreach ($feeStructures as $fs) {
            $gradesArray = explode(',', $fs->grades);
            if (in_array($studentClassroom, $gradesArray)) {
                $filtered[] = $fs;
            }
        }

        $base = 0.0;
        foreach ($filtered as $fs) {
            $base += (float) $fs->monthly_fee;
        }

        $this->calculatedBaseFee = $base;
        $this->calculatedFine = (float) $this->fine;
        $this->calculatedTotal = $base + (float) $this->fine - (float) $this->discount;
        $this->amount = max(0, $this->calculatedTotal);
    }

    /* ------------------------------ Pagamento ------------------------------ */

    public function collectFee()
    {
        $this->validate([
            'amount'      => 'required|numeric|min:0',
            'discount'    => 'numeric|min:0',
            'fine'        => 'numeric|min:0',
            'paymentMode' => 'required',
        ]);

        try {
            $paymentData = [
                'month'        => $this->month ?: date('F'),
                'year'         => $this->year ?: date('Y'),
                // IMPORTANTE: "amount" é o LÍQUIDO
                'amount'       => (float) $this->amount,
                'fine'         => (float) $this->fine,
                'discount'     => (float) $this->discount,
                'payment_mode' => $this->paymentMode,
                'paymentMode'  => $this->paymentMode, // compat.
                'pay_type'     => 'cash',
                'note'         => $this->note,
                'payment_date' => now(),
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentData
            );

            session()->flash('message', 'Pagamento registado com sucesso.');

            \Log::info('Admin fee payment collected', [
                'student_id'      => $this->studentId,
                'fee_assign_id'   => $feeAssign->id,
                'payment_mode'    => $this->paymentMode,
                'total_paid_net'  => $this->amount,
                'base_calculated' => $this->calculatedBaseFee,
                'fine_preserved'  => $feeAssign->fine,
                'discount_applied'=> $this->discount,
                'is_late_payment' => $this->isLatePayment,
                'month'           => $this->month,
                'year'            => $this->year,
            ]);

            $this->dispatchBrowserEvent('closeModal');
            $this->reset(['amount', 'discount', 'fine', 'paymentMode', 'note']);
            $this->calculateCurrentFee();
        } catch (\Exception $e) {
            \Log::error('Error collecting fee', [
                'student_id' => $this->studentId,
                'error'      => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao processar pagamento: ' . $e->getMessage());
        }
    }

    /* ------------------------------ Helpers ------------------------------ */

    /**
     * Divide um pagamento real do DB.
     * IMPORTANTE: No DB, "amount" é o LÍQUIDO (base + multa - desconto)
     * @return array [base, multa, desconto, liquido]
     */
//    private function splitPayment(?Fee_assign $p): array
//    {
//        if (!$p) return [0.0, 0.0, 0.0, 0.0];
//        $net      = (float) ($p->amount ?? 0);
//        $fine     = (float) ($p->fine ?? 0);
//        $discount = (float) ($p->discount ?? 0);
//        $base     = max(0, $net - $fine + $discount);
//        return [$base, $fine, $discount, $net];
//    }
    
    /**
     * Divide um pagamento do DB em [base, multa, desconto, liquido].
     * Decide automaticamente entre:
     *  - H1: amount = líquido (padrão do sistema)
     *  - H2: amount = base   (linhas antigas de 'Reference')
     */
    private function splitPayment(?Fee_assign $p): array
    {
        if (!$p) return [0.0, 0.0, 0.0, 0.0];

        $net      = (float) ($p->amount ?? 0);
        $fine     = max(0.0, (float) ($p->fine ?? 0));
        $discount = max(0.0, (float) ($p->discount ?? 0));

        // H1: amount é LÍQUIDO
        $base_h1 = max(0.0, $net - $fine + $discount);
        $net_h1  = $net;

        // H2: amount é BASE (caso legacy em Reference)
        $base_h2 = max(0.0, (float) ($p->amount ?? 0));
        $net_h2  = max(0.0, $base_h2 + $fine - $discount);

        $isReference = in_array(strtolower($p->pay_type ?? $p->paymentMode ?? ''), ['reference'], true);
        $hasFine     = $fine > 0.0;
        $eps         = 0.01;

        if ($isReference && $hasFine) {
            // Tenta estimar BASE esperada do mês para o aluno.
            try {
                $student = \App\Models\User::find($p->student_id);
                $classroom = $student->class->class
                    ?? $student->classroom->class
                    ?? null;

                $expectedBase = null;
                if ($classroom && ($p->month ?? null) && ($p->year ?? null)) {
                    $all = \App\Models\FeeStructure::where('active', 1)
                        ->whereRaw("FIND_IN_SET(?, grades)", [$classroom])
                        ->get();

                    $monthName = (string) $p->month;
                    $expectedBase = (float) $all->filter(function ($s) use ($monthName) {
                        return empty($s->months) || in_array($monthName, explode(',', $s->months));
                    })->sum('monthly_fee');
                }

                if ($expectedBase !== null && $expectedBase > 0) {
                    // escolhe a hipótese mais próxima da base esperada
                    $d1 = abs($base_h1 - $expectedBase);
                    $d2 = abs($base_h2 - $expectedBase);
                    if ($d2 + $eps < $d1) {
                        return [$base_h2, $fine, $discount, $net_h2];
                    }
                    return [$base_h1, $fine, $discount, $net_h1];
                }

                // Sem expectedBase: favorece H2 para Reference com multa (caso típico)
                return [$base_h2, $fine, $discount, $net_h2];

            } catch (\Throwable $e) {
                // Se der algum erro na estimativa, cai no fallback “favor H2”
                return [$base_h2, $fine, $discount, $net_h2];
            }
        }

        // Caso normal (não-Reference ou sem multa): H1
        return [$base_h1, $fine, $discount, $net_h1];
    }


    private function getFeeNamesForMonth($feeStructures, $month)
    {
        $names = [];
        foreach ($feeStructures as $structure) {
            if (empty($structure->months) || in_array($month, explode(',', $structure->months))) {
                $names[] = $structure->fee_name ?? 'Taxa Escolar';
            }
        }
        return empty($names) ? 'Taxa Escolar' : implode(', ', array_unique($names));
    }

    private function getRealFeeNameFromPayment($payment, $feeStructuresForMonth)
    {
        try {
            if (isset($payment->fee_group_id) && $payment->fee_group_id) {
                $feeGroup = \App\Models\FeeGroup::find($payment->fee_group_id);
                if ($feeGroup && $feeGroup->name) return $feeGroup->name;
            }
            if (isset($payment->fee_structure_id) && $payment->fee_structure_id) {
                $feeStructure = \App\Models\FeeStructure::find($payment->fee_structure_id);
                if ($feeStructure && $feeStructure->fee_name) return $feeStructure->fee_name;
            }

            $base = $this->splitPayment($payment)[0];
            foreach ($feeStructuresForMonth as $structure) {
                if (abs($structure->monthly_fee - $base) <= 0.01) {
                    return $structure->fee_name ?? 'Taxa Escolar';
                }
            }

            if (!empty($payment->note)) {
                $note = mb_strtolower($payment->note);
                if (str_contains($note, 'propina')) return 'Propina';
                if (str_contains($note, 'transporte')) return 'Transporte';
                if (str_contains($note, 'aliment')) return 'Alimentação';
                if (str_contains($note, 'material')) return 'Material';
            }
        } catch (\Exception $e) {
            \Log::warning('getRealFeeNameFromPayment fallback', ['error' => $e->getMessage()]);
        }

        $fallback = $feeStructuresForMonth->pluck('fee_name')->filter()->first();
        return $fallback ?: 'Taxa Escolar';
    }

    private function calculateDueDateForMonth($feeStructures, $monthName, $year)
    {
        foreach ($feeStructures as $structure) {
            $dueDateDay = (int) ($structure->payment_due_day ?? 5);
            $monthIndex = array_search($monthName, [
                'January','February','March','April','May','June',
                'July','August','September','October','November','December'
            ]) + 1;

            return str_pad($dueDateDay, 2, '0', STR_PAD_LEFT) . '/'
                 . str_pad($monthIndex, 2, '0', STR_PAD_LEFT) . '/'
                 . $year;
        }
        return 'N/A';
    }

    private function getPaymentReferenceSimple($month, $year)
    {
        static $cachedReference = null;

        if ($cachedReference === null) {
            $ref = \App\Models\PaymentReference::where('student_id', $this->studentId)
                ->where('fee_year', $year)
                ->whereNotNull('reference_number')
                ->first();
            $cachedReference = $ref ? $ref->reference_number : 'REF-' . $this->studentId;
        }

        $monthNumber = array_search($month, [
            'January','February','March','April','May','June',
            'July','August','September','October','November','December'
        ]) + 1;

        if (strlen($cachedReference) >= 11) {
            return substr($cachedReference, 0, 9) . str_pad($monthNumber, 2, '0', STR_PAD_LEFT);
        }

        return $cachedReference . str_pad($monthNumber, 2, '0', STR_PAD_LEFT);
    }

    /* ------------------------------ Export Excel ------------------------------ */

    public function exportExcel()
    {
        try {
            $data = $this->getExportDataComplete();

            return Excel::download(new class($data)
                implements
                    \Maatwebsite\Excel\Concerns\FromArray,
                    \Maatwebsite\Excel\Concerns\WithStyles,
                    \Maatwebsite\Excel\Concerns\WithTitle,
                    \Maatwebsite\Excel\Concerns\WithColumnWidths,
                    \Maatwebsite\Excel\Concerns\WithEvents {

                private $data;

                public function __construct($data) { $this->data = $data; }

                public function array(): array
                {
                    $rows = [];

                    // Título
                    $rows[] = ['COPMOZ - EXTRATO DE MENSALIDADES ' . $this->data['year']];
                    $rows[] = [''];

                    // Cabeçalho estudante
                    $rows[] = [
                        'Estudante: ' . $this->data['student']->name,
                        'ID: ' . $this->data['student']->student_id,
                        ($this->data['student']->class->class ?? 'N/A') . ' Classe - Turma: ' . ($this->data['student']->class->name ?? 'N/A'),
                        'Gerado: ' . now()->format('d/m/Y H:i'),
                    ];
                    $rows[] = [''];

                    // Resumo
                    $totalPago = $this->data['summary']['total_net_received'];
                    $totalEsperado = $this->data['summary']['total_annual_value'];
                    $percent = $totalEsperado > 0 ? round(($totalPago / $totalEsperado) * 100, 1) : 0;

                    $rows[] = [
                        '💰 Total Pago: MT ' . number_format($totalPago, 2, ',', '.'),
                        '📊 Total Esperado: MT ' . number_format($totalEsperado, 2, ',', '.'),
                        '📈 Progresso: ' . $percent . '%',
                        ''
                    ];
                    $rows[] = [''];

                    // Cabeçalho da tabela
                    $rows[] = [
                        'Período','Tipo de Taxa','Vencimento','Status',
                        'Valor Base (MT)','Multa (MT)','Desconto (MT)','Valor Pago (MT)','Data Pagamento'
                    ];

                    // Dados
                    foreach ($this->data['monthlyData'] as $m) {
                        $rows[] = [
                            $this->formatPeriodoCompacto($m['month'], $m['year']),
                            $this->simplificarTipoTaxa($m['fee_names']),
                            $m['due_date'],
                            $this->formatarStatusComEmoji($m['status']),
                            number_format($m['base_amount'], 2, ',', '.'),
                            number_format($m['fine_amount'], 2, ',', '.'),
                            number_format($m['discount_applied'], 2, ',', '.'),
                            number_format($m['paid_amount'], 2, ',', '.'),
                            $m['payment_date'] ?: '--',
                        ];
                    }

                    // Totais
                    $rows[] = [''];
                    $rows[] = [
                        '📊 TOTAIS:', '', '', '',
                        'MT ' . number_format($this->data['summary']['total_annual_value'], 2, ',', '.'),
                        'MT ' . number_format($this->data['summary']['total_fines_paid'], 2, ',', '.'),
                        'MT ' . number_format($this->data['summary']['total_discounts_given'], 2, ',', '.'),
                        'MT ' . number_format($this->data['summary']['total_net_received'], 2, ',', '.'),
                        ''
                    ];

                    // Estatísticas
                    $rows[] = [''];
                    $rows[] = [
                        '📈 ESTATÍSTICAS:',
                        '✅ Pagos: ' . $this->data['summary']['months_paid_count'] . '/' . max($this->data['summary']['months_with_fees'], $this->data['summary']['months_paid_count']) . ' meses',
                        '📊 Taxa de Cumprimento: ' . $this->data['summary']['payment_rate'] . '%',
                        ''
                    ];

                    return $rows;
                }

                /* ---------- Helpers de formatação de texto ---------- */
                private function formatPeriodoCompacto($month, $year) {
                    $map = [
                        'January'=>'Jan','February'=>'Fev','March'=>'Mar','April'=>'Abr','May'=>'Mai','June'=>'Jun',
                        'July'=>'Jul','August'=>'Ago','September'=>'Set','October'=>'Out','November'=>'Nov','December'=>'Dez'
                    ];
                    return ($map[$month] ?? $month) . ' ' . $year;
                }
                private function simplificarTipoTaxa($feeNames) {
                    if (!$feeNames) return 'Taxa Escolar';
                    $feeNames = trim($feeNames);
                    if (strlen($feeNames) > 20) {
                        if (stripos($feeNames, 'propina') !== false) return 'Propina';
                        if (stripos($feeNames, 'transporte') !== false) return 'Transporte';
                        if (stripos($feeNames, 'aliment') !== false) return 'Alimentação';
                        if (stripos($feeNames, 'material') !== false) return 'Material';
                        if (stripos($feeNames, 'matric') !== false) return 'Matrícula';
                        return mb_substr($feeNames, 0, 17) . '...';
                    }
                    return $feeNames;
                }
                private function formatarStatusComEmoji($status) {
                    return $status === 'Pago' ? '✅ Pago'
                        : ($status === 'Em Atraso' ? '⏰ Em Atraso'
                        : ($status === 'Não Pago' ? '⏳ Pendente' : '❓ '.$status));
                }

                /* ---------- Título/colunas ---------- */
                public function title(): string { return 'Mensalidades ' . $this->data['year']; }
                public function columnWidths(): array {
                    return ['A'=>18,'B'=>28,'C'=>16,'D'=>18,'E'=>16,'F'=>14,'G'=>14,'H'=>16,'I'=>22];
                }

                /* ---------- Estilos base (linhas específicas) ---------- */
                public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
                {
                    return [
                        1 => ['font'=>['bold'=>true,'size'=>14,'color'=>['rgb'=>'1565C0']], 'alignment'=>['horizontal'=>'center']],
                        3 => ['font'=>['bold'=>true]],
                        5 => ['font'=>['bold'=>true,'color'=>['rgb'=>'2E7D32']]],
                        7 => ['font'=>['bold'=>true,'color'=>['rgb'=>'FFFFFF']]],
                    ];
                }

                /* ---------- Pós-processamento: cores, bordas, zebra, alinhamentos ---------- */
                public function registerEvents(): array
                {
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function(\Maatwebsite\Excel\Events\AfterSheet $event) {
                            $sheet = $event->sheet->getDelegate();

                            // Mescla título + congela o cabeçalho
                            $sheet->mergeCells('A1:I1');
                            $sheet->getRowDimension(1)->setRowHeight(24);
                            $sheet->freezePane('A8');

                            // Linha do cabeçalho da tabela (7): fundo azul
                            $sheet->getStyle('A7:I7')->getFill()->setFillType('solid')->getStartColor()->setRGB('1565C0');
                            $sheet->getStyle('A7:I7')->getAlignment()->setHorizontal('center')->setVertical('center');
                            $sheet->getRowDimension(7)->setRowHeight(20);

                            // Faixa de dados
                            $dataRowStart = 8;
                            $rows = count($this->data['monthlyData']);
                            $dataRowEnd = $dataRowStart + max(0, $rows - 1);

                            if ($rows > 0) {
                                // Zebra
                                for ($r = $dataRowStart; $r <= $dataRowEnd; $r++) {
                                    if ($r % 2 === 0) {
                                        $sheet->getStyle("A{$r}:I{$r}")->getFill()->setFillType('solid')->getStartColor()->setRGB('F7F9FC');
                                    }
                                }

                                // Alinhamentos
                                $sheet->getStyle("A{$dataRowStart}:B{$dataRowEnd}")->getAlignment()->setHorizontal('left');
                                $sheet->getStyle("C{$dataRowStart}:D{$dataRowEnd}")->getAlignment()->setHorizontal('center');
                                $sheet->getStyle("E{$dataRowStart}:H{$dataRowEnd}")->getAlignment()->setHorizontal('right');
                                $sheet->getStyle("A{$dataRowStart}:I{$dataRowEnd}")->getAlignment()->setVertical('center');

                                // Bordas finas na tabela
                                $sheet->getStyle("A7:I{$dataRowEnd}")
                                      ->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN)
                                      ->getColor()->setRGB('CFD8DC');

                                // Cores por status (coluna D)
                                foreach ($this->data['monthlyData'] as $i => $monthData) {
                                    $row = $dataRowStart + $i;
                                    $status = $monthData['status'];
                                    $fill = 'FFFFFF';
                                    if ($status === 'Pago') $fill = 'C8E6C9';          // verde claro
                                    elseif ($status === 'Em Atraso') $fill = 'FFCDD2'; // vermelho claro
                                    elseif ($status === 'Não Pago') $fill = 'FFF9C4'; // amarelo claro
                                    $sheet->getStyle("D{$row}")->getFill()->setFillType('solid')->getStartColor()->setRGB($fill);
                                    $sheet->getStyle("D{$row}")->getFont()->setBold(true);
                                    $sheet->getStyle("D{$row}")->getAlignment()->setHorizontal('center');
                                }
                            }

                            // Linha de totais (2 linhas abaixo dos dados)
                            $totalsRow = ($rows > 0 ? $dataRowEnd : 7) + 2;
                            $sheet->getStyle("A{$totalsRow}:I{$totalsRow}")
                                  ->getFill()->setFillType('solid')->getStartColor()->setRGB('ECEFF1'); // cinza claro
                            $sheet->getStyle("A{$totalsRow}:I{$totalsRow}")->getFont()->setBold(true);
                            $sheet->getStyle("E{$totalsRow}:H{$totalsRow}")->getAlignment()->setHorizontal('right');

                            // Pequeno destaque na linha de estatísticas (2 linhas abaixo de totais)
                            $statsRow = $totalsRow + 2;
                            $sheet->getStyle("A{$statsRow}:D{$statsRow}")->getFont()->setBold(true)->getColor()->setRGB('2E7D32');
                        }
                    ];
                }
            }, "mensalidades_{$this->selectedStudent->student_id}_{$this->selectedYear}.xlsx");

        } catch (\Exception $e) {
            \Log::error('Excel export failed', ['error' => $e->getMessage()]);
            session()->flash('error', 'Erro ao exportar Excel: ' . $e->getMessage());
            return back();
        }
    }


    private function getExportDataComplete()
    {
        $currentYear = $this->selectedYear ?? now()->year;
        $months = ['January','February','March','April','May','June','July','August','September','October','November','December'];

        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $allFeeStructures = collect();
        if ($studentClassroom) {
            $allFeeStructures = FeeStructure::where('active', 1)
                ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                ->get();
        }

        $payments = Fee_assign::where('student_id', $this->studentId)
            ->where('year', $currentYear)
            ->get()
            ->keyBy('month');

        $monthlyData = [];
        $summary = [
            'total_annual_value'    => 0.0,
            'total_paid'            => 0.0, // base recebida
            'total_fines_paid'      => 0.0,
            'total_discounts_given' => 0.0,
            'total_net_received'    => 0.0,
            'months_with_fees'      => 0,
            'months_paid_count'     => 0,
        ];

        foreach ($months as $monthName) {
            $payment = $payments->get($monthName);
            $isPaid  = (bool) $payment;

            $feeStructuresForMonth = $allFeeStructures->filter(function($s) use ($monthName) {
                return empty($s->months) || in_array($monthName, explode(',', $s->months));
            });

            $baseAmountForMonth = (float) $feeStructuresForMonth->sum('monthly_fee');
            if (!$isPaid && $baseAmountForMonth <= 0) continue;

            if ($isPaid) {
                [$baseAmount, $fineAmount, $discountAmount, $netPaidAmount] = $this->splitPayment($payment);
                $paymentDate = $payment->created_at->format('d/m/Y H:i');
                $status = 'Pago';
                $feeNames = $this->getRealFeeNameFromPayment($payment, $feeStructuresForMonth);
                $dueDate = $this->calculateDueDateForMonth($feeStructuresForMonth, $monthName, $currentYear);
            } else {
                $baseAmount = $baseAmountForMonth;
                $dueDate = $this->calculateDueDateForMonth($feeStructuresForMonth, $monthName, $currentYear);

                $isOverdue = false;
                if ($dueDate !== 'N/A') {
                    try { $isOverdue = now()->gt(Carbon::createFromFormat('d/m/Y', $dueDate)); }
                    catch (\Exception $e) { $isOverdue = false; }
                }

                // Multa prevista (somente para exibição)
                $fineAmount = 0.0;
                if ($isOverdue) {
                    foreach ($feeStructuresForMonth as $structure) {
                        if (($structure->penalty_type ?? null) === 'fixed') {
                            $fineAmount += (float) ($structure->late_penalty_percentage ?? 0);
                        } else {
                            $fineAmount += ((float) $structure->monthly_fee) * ((float) ($structure->late_penalty_percentage ?? 0)) / 100;
                        }
                    }
                }

                $feeNames = $feeStructuresForMonth->pluck('fee_name')->filter()->unique()->implode(', ');
                if (!$feeNames) $feeNames = 'Taxa Escolar';

                $discountAmount = 0.0;
                $netPaidAmount  = 0.0;
                $paymentDate    = '--';
                $status         = $isOverdue ? 'Em Atraso' : 'Não Pago';
            }

            $monthlyData[] = [
                'month'            => $monthName,
                'year'             => $currentYear,
                'reference'        => $this->getPaymentReferenceSimple($monthName, $currentYear),
                'fee_names'        => $feeNames,
                'due_date'         => $dueDate,
                'status'           => $status,
                'base_amount'      => $baseAmount,       // BASE
                'fine_amount'      => $fineAmount,       // multa (real ou prevista)
                'discount_applied' => $discountAmount,
                'paid_amount'      => $netPaidAmount,    // LÍQUIDO
                'payment_date'     => $paymentDate,
                'is_paid'          => $isPaid,
                'is_overdue'       => $status === 'Em Atraso',
            ];

            if ($baseAmount > 0) {
                $summary['total_annual_value'] += $baseAmount; // esperado = soma das bases
                $summary['months_with_fees']++;
            }

            if ($isPaid) {
                $summary['total_paid']            += $baseAmount;    // base recebida
                $summary['total_fines_paid']      += $fineAmount;
                $summary['total_discounts_given'] += $discountAmount;
                $summary['total_net_received']    += $netPaidAmount; // líquido
                $summary['months_paid_count']++;
            }
        }

        $summary['payment_rate'] = $summary['months_with_fees'] > 0
            ? round(($summary['months_paid_count'] / $summary['months_with_fees']) * 100, 1)
            : 0;

        return [
            'monthlyData' => $monthlyData,
            'summary'     => $summary,
            'student'     => $this->selectedStudent,
            'year'        => $currentYear,
        ];
    }

    /* ------------------------------ Export PDF ------------------------------ */

    public function exportPdf()
    {
        try {
            $data = $this->getExportData();

            $pdf = Pdf::loadView('exports.student-finance-pdf', [
                'student'      => $this->selectedStudent,
                'year'         => $this->selectedYear,
                'data'         => $data,
                'generated_at' => now()->format('d/m/Y H:i'),
            ]);

            return $pdf->download("extrato_financeiro_{$this->selectedStudent->student_id}_{$this->selectedYear}.pdf");
        } catch (\Exception $e) {
            \Log::error('PDF export failed', ['error' => $e->getMessage()]);
            session()->flash('error', 'Erro ao exportar PDF: ' . $e->getMessage());
        }
    }

    private function getExportData()
    {
        $currentYear = $this->selectedYear ?? now()->year;
        $months = ['January','February','March','April','May','June','July','August','September','October','November','December'];

        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $filteredFeeStructures = [];
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)->get();
            foreach ($feeStructures as $fs) {
                $grades = explode(',', $fs->grades);
                if (in_array($studentClassroom, $grades)) $filteredFeeStructures[] = $fs;
            }
        }

        $exportRows = [];
        $summary = [
            'total_annual_value'    => 0.0,
            'total_paid'            => 0.0, // base recebida
            'total_due'             => 0.0,
            'total_overdue'         => 0.0,
            'total_fines_paid'      => 0.0,
            'total_discounts_given' => 0.0,
            'total_net_received'    => 0.0,
            'months_with_fees'      => 0,
            'months_paid_count'     => 0,
        ];

        foreach ($months as $monthName) {
            try {
                $calc = $this->feeCalculationService->calculateFeeForStudent(
                    $this->selectedStudent, $monthName, $currentYear
                );
                if ($calc['base_amount'] <= 0) continue;

                $existing = $this->feeCalculationService->getPaymentForPeriod(
                    $this->selectedStudent, $monthName, $currentYear
                );

                $isPaid = (bool) $existing;
                [$basePaid, $finePaid, $discount, $netPaid] = $this->splitPayment($existing);

                $fineToShow = $isPaid ? $finePaid : (float) $calc['fine_amount'];
                $status = $isPaid ? 'Pago' : ($calc['is_late_payment'] ? 'Em Atraso' : 'Não Pago');
                $feeNames = $this->getFeeNamesForMonth($filteredFeeStructures, $monthName);
                $dueDate = $calc['due_date'] ? $calc['due_date']->format('d/m/Y') : 'N/A';
                $paymentDate = $isPaid ? $existing->created_at->format('d/m/Y H:i') : '--';

                // linhas para o PDF/Excel simples
                $exportRows[] = [
                    $monthName . ' ' . $currentYear,
                    $feeNames,
                    $dueDate,
                    $status,
                    number_format($calc['base_amount'], 2, ',', '.'),
                    number_format($fineToShow, 2, ',', '.'),
                    number_format($discount, 2, ',', '.'),
                    number_format($netPaid, 2, ',', '.'),
                    $paymentDate,
                ];

                // totais (esperado = soma das bases)
                $summary['total_annual_value'] += (float) $calc['base_amount'];
                $summary['months_with_fees']++;

                if ($isPaid) {
                    $summary['total_paid']            += (float) $basePaid; // base recebida
                    $summary['total_fines_paid']      += (float) $finePaid;
                    $summary['total_discounts_given'] += (float) $discount;
                    $summary['total_net_received']    += (float) $netPaid;
                    $summary['months_paid_count']++;
                } else {
                    if ($calc['is_late_payment']) {
                        $summary['total_overdue'] += (float) $calc['base_amount'];
                    } else {
                        $summary['total_due']     += (float) $calc['base_amount'];
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error preparing export data', [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'error' => $e->getMessage(),
                ]);
                continue;
            }
        }

        $summary['payment_rate'] = $summary['months_with_fees'] > 0
            ? round(($summary['months_paid_count'] / $summary['months_with_fees']) * 100, 1)
            : 0;

        return [
            'rows'    => $exportRows,
            'summary' => $summary,
            'student' => $this->selectedStudent,
            'year'    => $currentYear,
        ];
    }

    /* ------------------------------ UI helpers ------------------------------ */

    public function openModal($feeAssignId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeAssignId;
        $this->month = (string) $month;
        $this->year  = (int) $year;

        $this->calculateCurrentFee();

        $this->dispatchBrowserEvent('openModal');
        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
    }

    public function setMonth($month) { $this->month = $month; $this->calculateCurrentFee(); }
    public function setYear($year)   { $this->year  = $year;  $this->calculateCurrentFee(); }
    public function addFine($amount) { $this->fine += $amount; $this->calculateCurrentFee(); }
    public function addDiscount($amount) { $this->discount += $amount; $this->calculateCurrentFee(); }
    public function resetCalculation()
    {
        $this->fine = 0;
        $this->discount = 0;
        $this->month = null;
        $this->year = null;
        $this->calculateCurrentFee();
    }

    public function checkExistingPayment($month, $year)
    {
        return $this->feeCalculationService->hasPaymentForPeriod(
            $this->selectedStudent, $month, $year
        );
    }

    public function getCalculationDetailsProperty()
    {
        return [
            'base_fee'             => $this->calculatedBaseFee,
            'calculated_auto_fine' => max(0, $this->calculatedFine - (float) $this->fine),
            'manual_fine'          => (float) $this->fine,
            'total_fine'           => $this->calculatedFine,
            'discount'             => (float) $this->discount,
            'total'                => $this->calculatedTotal,
            'is_late'              => $this->isLatePayment,
            'due_date'             => $this->dueDate?->format('d/m/Y'),
            'month'                => $this->month ?: 'Current',
            'year'                 => $this->year ?: date('Y'),
            'preservation_note'    => 'As multas são preservadas no histórico para auditoria completa',
        ];
    }

    /* ------------------------------ Render ------------------------------ */

    public function render()
    {
        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $filteredFeeStructures = [];
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)->get();
            foreach ($feeStructures as $fs) {
                $gradesArray = explode(',', $fs->grades);
                if (in_array($studentClassroom, $gradesArray)) $filteredFeeStructures[] = $fs;
            }
        }

        $currentYear = $this->selectedYear ?? date('Y');
        $months = ['January','February','March','April','May','June','July','August','September','October','November','December'];

        $monthlyData = [];

        // Totais compatibilidade
        $totalExpected = 0.0; // AGORA = soma das BASES
        $totalPaid     = 0.0; // líquido recebido
        $totalPending  = 0.0; // base pendente
        $totalOverdue  = 0.0; // base atrasada

        // Novos totais (widgets)
        $totalAnnualValue   = 0.0;
        $totalBasePaid      = 0.0;
        $totalDue           = 0.0;
        $totalOverdueBase   = 0.0;
        $totalFinesPaid     = 0.0;
        $totalDiscountsGiven= 0.0;
        $totalNetReceived   = 0.0;
        $monthsWithFees     = 0;
        $monthsPaidCount    = 0;

        foreach ($months as $monthName) {
            try {
                $calculation = $this->feeCalculationService->calculateFeeForStudent(
                    $this->selectedStudent, $monthName, $currentYear
                );
                if ($calculation['base_amount'] <= 0) continue;

                $existingPayment = $this->feeCalculationService->getPaymentForPeriod(
                    $this->selectedStudent, $monthName, $currentYear
                );

                $isPaid = (bool) $existingPayment;
                [$baseAmountPaid, $fineAmountPaid, $discountAmount, $netPaidAmount] = $this->splitPayment($existingPayment);

                $fineAmountToShow = $isPaid ? $fineAmountPaid : (float) $calculation['fine_amount'];
                $status = $isPaid ? 'Pago' : ($calculation['is_late_payment'] ? 'Em Atraso' : 'Não Pago');

                $balanceAmount = $isPaid ? 0.0 : (float) $calculation['base_amount'];

                $monthlyData[] = [
                    'month'            => $monthName,
                    'year'             => $currentYear,
                    'fee_names'        => $this->getFeeNamesForMonth($filteredFeeStructures, $monthName),
                    'due_date'         => $calculation['due_date'] ? $calculation['due_date']->format('d/m/Y') : 'N/A',
                    'status'           => $status,
                    'base_amount'      => (float) $calculation['base_amount'],  // BASE (igual para todos)
                    'fine_amount'      => $fineAmountToShow,                    // multa (real ou prevista)
                    'total_expected'   => (float) $calculation['base_amount'],  // ESPERADO = BASE
                    'discount_applied' => $discountAmount,
                    'paid_amount'      => $netPaidAmount,                       // LÍQUIDO
                    'balance_amount'   => $balanceAmount,
                    'payment_date'     => $isPaid ? $existingPayment->created_at->format('d/m/Y H:i') : 'N/A',
                    'is_paid'          => $isPaid,
                    'is_overdue'       => (bool) $calculation['is_late_payment'],
                    'existing_payment' => $existingPayment,
                    'data_status'      => $isPaid ? 'paid' : ($calculation['is_late_payment'] ? 'overdue' : 'pending'),
                ];

                // Totais (widgets)
                $monthsWithFees++;
                $totalAnnualValue += (float) $calculation['base_amount'];

                if ($isPaid) {
                    $totalBasePaid       += $baseAmountPaid;
                    $totalFinesPaid      += $fineAmountPaid;
                    $totalDiscountsGiven += $discountAmount;
                    $totalNetReceived    += $netPaidAmount;
                    $monthsPaidCount++;
                } else {
                    if ($calculation['is_late_payment']) {
                        $totalOverdueBase += (float) $calculation['base_amount'];
                    } else {
                        $totalDue += (float) $calculation['base_amount'];
                    }
                }

                // Totais compatibilidade (esperado/pago/pendente/atrasado)
                $totalExpected += (float) $calculation['base_amount']; // apenas base
                if ($isPaid) {
                    $totalPaid += $netPaidAmount; // líquido recebido
                } else {
                    if ($calculation['is_late_payment']) {
                        $totalOverdue += (float) $calculation['base_amount'];
                    } else {
                        $totalPending += (float) $calculation['base_amount'];
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error calculating month data', [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'student_id' => $this->studentId,
                    'error' => $e->getMessage(),
                ]);

                $monthlyData[] = [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'fee_names' => 'Taxa Escolar',
                    'due_date' => 'N/A',
                    'status' => 'Erro',
                    'base_amount' => 0,
                    'fine_amount' => 0,
                    'total_expected' => 0,
                    'discount_applied' => 0,
                    'paid_amount' => 0,
                    'balance_amount' => 0,
                    'payment_date' => 'N/A',
                    'is_paid' => false,
                    'is_overdue' => false,
                    'existing_payment' => null,
                    'data_status' => 'error',
                ];
            }
        }

        $paymentRate = $monthsWithFees > 0 ? round(($monthsPaidCount / $monthsWithFees) * 100, 1) : 0;

        $fees = Fee_assign::where('student_id', $this->studentId)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($fee) {
                $fee->total_expected   = $fee->amount + $fee->fine - $fee->discount;
                $fee->has_fine         = $fee->fine > 0;
                $fee->fine_percentage  = $fee->amount > 0 ? round(($fee->fine / $fee->amount) * 100, 1) : 0;
                $fee->payment_status   = 'complete';
                return $fee;
            });

        $discounts = discount::where('student_id', $this->studentId)->get();

        return view('livewire.admin.fee.fee-collection', [
            'student'          => $this->selectedStudent,
            'fees'             => $fees,
            'fee'              => $this->calculatedTotal,
            'feestructures'    => $filteredFeeStructures,
            'discounts'        => $discounts,
            'calculatedBaseFee'=> $this->calculatedBaseFee,
            'calculatedFine'   => $this->calculatedFine,
            'calculatedTotal'  => $this->calculatedTotal,
            'isLatePayment'    => $this->isLatePayment,
            'dueDate'          => $this->dueDate,

            // Tabela mensal
            'monthlyData' => $monthlyData,

            // Resumo compatível
            'yearSummary' => [
                'total_expected' => $totalExpected, // soma das bases
                'total_paid'     => $totalPaid,     // líquido recebido
                'total_pending'  => $totalPending,  // base pendente
                'total_overdue'  => $totalOverdue,  // base atrasada
                'current_year'   => $currentYear,
            ],

            'selectedYear' => $this->selectedYear,

            // Widgets
            'widgetData' => [
                'total_annual_value'    => $totalAnnualValue,
                'total_base_paid'       => $totalBasePaid,
                'total_net_received'    => $totalNetReceived,
                'total_due'             => $totalDue,
                'total_overdue_base'    => $totalOverdueBase,
                'total_fines_paid'      => $totalFinesPaid,
                'total_discounts_given' => $totalDiscountsGiven,
                'payment_rate'          => $paymentRate,
                'months_with_fees'      => $monthsWithFees,
                'months_paid_count'     => $monthsPaidCount,
            ],

            'analysisData' => [
                'total_students_in_class' => 1,
                'average_payment_delay'   => 0,
                'most_delayed_month'      => null,
                'payment_trend'           => 'stable',
            ],

            'uiConfig' => [
                'show_advanced_stats' => true,
                'enable_bulk_actions' => false,
                'default_payment_mode'=> 'Cash',
                'currency_symbol'     => 'MT',
                'date_format'         => 'd/m/Y',
                'enable_debug_mode'   => app()->environment('local'),
            ],

            'reportMetadata' => [
                'generated_at'          => now(),
                'generated_by'          => auth()->user()->name ?? 'Sistema',
                'academic_year'         => $currentYear,
                'student_class'         => $studentClassroom,
                'total_months_analyzed' => count($months),
                'months_with_data'      => count(array_filter($monthlyData, fn($m) => $m['base_amount'] > 0)),
                'data_quality_score'    => $paymentRate,
            ],
        ]);
    }
}

