<?php

namespace App\Http\Livewire\Admin\Notifications;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\User;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Livewire\Component;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Exception\Messaging\NotFound;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Note\Note;

class AddPushNotification extends Component
{
    use LivewireAlert;

    public $type = 'all';
    public $class_id;
    public $student_id;
    public $message;
    public $student_type;
    public $title;

    protected function initializeFirebase()
    {
        // Move credentials to .env file for security
        return (new Factory)
            ->withServiceAccount([
                "type" => "service_account",
                "project_id" => env('FIREBASE_PROJECT_ID'),
                "private_key_id" => env('FIREBASE_PRIVATE_KEY_ID'),
                "private_key" => str_replace("\\n", "\n", env('FIREBASE_PRIVATE_KEY')),
                "client_email" => env('FIREBASE_CLIENT_EMAIL'),
                "client_id" => env('FIREBASE_CLIENT_ID'),
                "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
                "token_uri" => "https://oauth2.googleapis.com/token",
                "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
                "client_x509_cert_url" => env('FIREBASE_CLIENT_X509_CERT_URL'),
                "universe_domain" => "googleapis.com"
            ])
            ->createMessaging();
    }

    protected function sendNotificationToToken($user, $token, $messaging)
    {
        try {
            $notification = Notification::create($this->title, $this->message);
            
            $message = CloudMessage::withTarget('token', $token)
                ->withNotification($notification)
                ->withHighestPossiblePriority()
                ->withDefaultSounds();
                
            $messaging->send($message);
            
            Log::channel('log_notifications')->info("Notification sent successfully to: {$user->name} - Token: $token");
            return true;
            
        } catch (NotFound $e) {
            // Token not found (unregistered or invalid) - delete it
            $user->deviceTokens()->where('token', $token)->delete();
            Log::channel('log_notifications')->warning("Token not found/deleted for {$user->name}: {$token}. Error: {$e->getMessage()}");
        } catch (\Kreait\Firebase\Exception\MessagingException $e) {
            Log::channel('log_notifications')->error("Messaging error for {$user->name}: {$e->getMessage()}");
        } catch (\Exception $e) {
            Log::channel('log_notifications')->error("General error for {$user->name}: {$e->getMessage()}");
        }
        
        return false;
    }

    public function submit()
    {
        $this->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:all,class,student',
            'class_id' => 'required_if:type,class',
            'student_id' => 'required_if:type,student',
        ]);

        $messaging = $this->initializeFirebase();
        $successCount = 0;
        $failureCount = 0;

        try {
            if ($this->type == 'all') {
                $users = User::whereHas('deviceTokens')->get();
                
                Log::channel('log_notifications')->info("Sending to all users. Total with device tokens: " . $users->count());
                
                foreach ($users as $user) {
                    $tokens = $user->deviceTokens()->pluck('token');
                    
                    foreach ($tokens as $token) {
                        if ($this->sendNotificationToToken($user, $token, $messaging)) {
                            $successCount++;
                        } else {
                            $failureCount++;
                        }
                    }
                }

            } elseif ($this->type == 'class') {
                $users = User::where('classroom_id', $this->class_id)
                    ->whereHas('deviceTokens')
                    ->get();
                    
                foreach ($users as $user) {
                    $tokens = $user->deviceTokens()->pluck('token');
                    
                    foreach ($tokens as $token) {
                        if ($this->sendNotificationToToken($user, $token, $messaging)) {
                            $successCount++;
                        } else {
                            $failureCount++;
                        }
                    }
                }
                
            } elseif ($this->type == 'student') {
                $user = User::find($this->student_id);
                
                if ($user) {
                    $tokens = $user->deviceTokens()->pluck('token');
                    
                    foreach ($tokens as $token) {
                        if ($this->sendNotificationToToken($user, $token, $messaging)) {
                            $successCount++;
                        } else {
                            $failureCount++;
                        }
                    }
                }
            }
            
            $message = "Push notifications sent successfully. Success: {$successCount}, Failures: {$failureCount}";
            $this->alert('success', $message);
            Log::channel('log_notifications')->info($message);
            
        } catch (\Exception $e) {
            $errorMsg = "Failed to send notifications: " . $e->getMessage();
            $this->alert('error', $errorMsg);
            Log::channel('log_notifications')->error($errorMsg);
        }

        $this->reset(['type', 'class_id', 'student_id', 'message', 'title']);
    }

    public function render()
    {
        Note::createSystemNotification(Admin::class, 'Notificação', 'Notificação enviada com sucesso.');

        // Mantém a lógica original de carregar estudantes
        if(!empty($this->type)) {
            return view('livewire.admin.notifications.add-notifications', [
                'classrooms' => Classroom::query()->orderBy('class')->get(),
                'students' => User::when($this->class_id, function($query) {
                    return $query->where('classroom_id', $this->class_id);
                })->orderBy('name')->get(),
            ]);
        } else {
            return view('livewire.admin.notifications.add-notifications', [
                'classrooms' => Classroom::query()->orderBy('class')->get(),
            ]);
        }
    }
}
