<?php

namespace App\Http\Livewire\Admin\Reports;

use App\Models\Fee_assign;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Livewire\Component;

class FeeReportController extends Component
{
    /**
     * Display the Fees Collection Report
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function feesCollectionReport(Request $request)
    {
        // Set the default start and end dates
        $startDate = $request->input('start_date', Carbon::now()->subMonths(3)->format('Y-m-d')); // Last 3 months default
        $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d')); // Current date default

        // Fetch the data based on filters
        $feesCollection = Fee_assign::with('user')
            ->when($request->input('class'), function($query) use ($request) {
                return $query->where('class', $request->input('class'));
            })
            ->when($request->input('section'), function($query) use ($request) {
                return $query->where('section', $request->input('section'));
            })
            ->when($request->input('fee_type'), function($query) use ($request) {
                return $query->where('fee_type', $request->input('fee_type'));
            })
            ->when($request->input('pay_type'), function($query) use ($request) {
                return $query->where('pay_type', $request->input('pay_type'));
            })
            ->whereBetween('pay_day', [$startDate, $endDate])
            ->get();

        return view('livewire.admin.reports.fees_collection_report', compact('feesCollection'));
    }

    /**
     * Display the Daily Collection Report
     *
     * @return \Illuminate\View\View
     */
    public function dailyCollectionReport()
    {
        // Get data for today
        $dailyFees = Fee_assign::with('user')
            ->whereDate('pay_day', Carbon::today()->format('Y-m-d'))
            ->get();

        return view('livewire.admin.reports.daily_collection_report', compact('dailyFees'));
    }

    /**
     * Display the Balance Fees Report
     *
     * @return \Illuminate\View\View
     */
    public function balanceFeesReport()
    {
        // Fetch records where status is 'unpaid'
        $balanceFees = Fee_assign::with('user')
            ->where('status', 'unpaid')
            ->get();

        return view('livewire.admin.reports.balance_fees_report', compact('balanceFees'));
    }

    /**
     * Display the Online Fees Collection Report
     *
     * @return \Illuminate\View\View
     */
    public function onlineFeesCollectionReport()
    {
        // Fetch records where pay_type is 'Online'
        $onlineFees = Fee_assign::with('user')
            ->where('pay_type', 'Online')
            ->get();

        return view('livewire.admin.reports.online_fees_collection_report', compact('onlineFees'));
    }

    /**
     * Display the Fees Statement Report for a particular student
     *
     * @param Request $request
     * @param $studentId
     * @return \Illuminate\View\View
     */
    public function feesStatement(Request $request, $studentId)
    {
        // Fetch the fees related to a particular student
        $studentFees = Fee_assign::with('user')
            ->where('student_id', $studentId)
            ->get();

        return view('livewire.admin.reports.fees_statement', compact('studentFees'));
    }

    /**
     * Display the Income Report (total fees collected)
     *
     * @return \Illuminate\View\View
     */
    public function incomeReport()
    {
        // Calculate total income based on the fees collected
        $income = Fee_assign::sum('amount');

        return view('livewire.admin.reports.income_report', compact('income'));
    }

    /**
     * Display the Expense Report
     *
     * @return \Illuminate\View\View
     */
    public function expenseReport()
    {
        // You can integrate expense data as per your application needs
        // For now, let's return an empty report
        $expenses = []; // Fetch expenses from a table if you have one
        return view('livewire.admin.reports.expense_report', compact('expenses'));
    }

    /**
     * Display the Income Expense Balance Report
     *
     * @return \Illuminate\View\View
     */
    public function incomeExpenseBalanceReport()
    {
        // You can calculate the balance between income and expenses here
        // If you have an expense table, you can fetch data and subtract from the income
        $income = Fee_assign::sum('amount');
        $expenses = []; // Placeholder for expense data

        $balance = $income - array_sum($expenses);

        return view('livewire.admin.reports.income_expense_balance_report', compact('income', 'expenses', 'balance'));
    }
}
