<?php

namespace App\Http\Livewire\Admin\Teachers;

use App\Models\Admin;
use App\Models\Subject;
use App\Models\TeacherSubject;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;
use App\Models\Classroom;

class AssignMultipleSubject extends Component
{
    use FindGuard, LivewireAlert;

    public $teacher_ids = [];
    public $teacher_id = [];
    public $subject_ids = [];
    public $classroom_ids = [];
    public $classroom_id = [];
    public $selectedSubjects = [];
    public $classrooms = [];
    public $class;
    public $validatedData;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    protected array $rules = [
        'teacher_id' => ['required', 'exists:admins,id'],
        'class' => ['required', 'numeric', 'between:1,12'],
        'classroom_id' => ['required', 'array'],
        'classroom_id.*' => ['required', 'exists:classrooms,id'],
        'selectedSubjects' => ['required', 'array'],
        'selectedSubjects.*' => ['required', 'exists:subjects,id'],
    ];
    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
            // If the 'class' property changes, update the classrooms
    if ($propertyName === 'class') {
        // Fetch classrooms based on the selected class
        $this->classrooms = Classroom::where('class', $this->class)->get();
        
    }
    }

    public function submit()
    {
        
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }


    public function confirmed()
    {
        // Ensure teacher_id and classroom_id are arrays
        $teacherIds = is_array($this->teacher_id) ? $this->teacher_id : [$this->teacher_id];
        $classroomIds = is_array($this->classroom_id) ? $this->classroom_id : [$this->classroom_id];

        
        foreach ($teacherIds as $teacher_id) {
            foreach ($this->selectedSubjects as $subject_id) {
                foreach ($classroomIds as $classroom_id) {
                    $check = TeacherSubject::query()
                        ->where('teacher_id', $teacher_id)
                        ->where('classroom_id', $classroom_id)
                        ->where('class', $this->class)
                        ->where('subject_id', $subject_id)
                        ->exists();
    
                    if ($check) {
                        $this->alert('error', 'Professor já tem disciplina atribuida.');
                        return back();
                    }
    
                    $teacherObject = [
                        'teacher_id' => $teacher_id,
                        'subject_id' => $subject_id,
                        'classroom_id' => $classroom_id,
                        'class' => $this->class,
                    ];
    
                    TeacherSubject::create($teacherObject);
    
                    Note::createSystemNotification(
                        Admin::class,
                        'Nova atribuição de disciplina',
                        'Disciplina atribuída ao professor com sucesso ' . Admin::find($teacher_id)->name
                    );
    
                    $this->alert('success', 'Disciplina atribuída ao professor com sucesso ' . Admin::find($teacher_id)->name);
                    $this->reset();
                }
            }
        }
    
        $this->emit('noteAdded');
        $this->alert('success', 'Disciplinas atribuídas aos professores com sucesso.');
        $this->reset();
    }


    public function cancelled()
    {
        $this->alert('error', 'Operação cancelada.');
    }

    public function render()
    {
        
        return view('livewire.admin.teachers.assign-multiple-subject', [
            'subjects' => Subject::query()->orderBy('name')->get(),
            'teachers' => Admin::query()->orderBy('name')->where('role', 2)->get(),
            'classes' => Classroom::query()->where(function ($query) {
                $query->orWhere('class', $this->class);
            })->get(),
        ]);
    }
}

